#   Copyright 2020 Atos
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [SLICE-02] Shared Network Slicing.

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nst_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsi_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/connectivity_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot

Force Tags   slice_02   cluster_slices   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***

# NS and VNF descriptor package files
${vnfd1_pkg}   slice_basic_vnf
${vnfd2_pkg}   slice_basic_middle_vnf
${nsd1_pkg}   slice_basic_ns
${nsd2_pkg}   slice_basic_middle_ns
${nst}   slice_basic_nst/slice_basic_nst.yaml
${nst2}   slice_basic_nst/slice_basic_2nd_nst.yaml

# Instance names
${slice_name}   slicebasic
${slice2_name}   sliceshared
${middle_ns_name}   slicebasic.slice_basic_nsd_2

# Descriptor names
${nst_name}   slice_basic_nst
${nst2_name}   slice_basic_nst2
${vnfd1_name}   slice_basic_vnf
${vnfd2_name}   slice_basic_middle_vnf
${nsd1_name}   slice_basic_ns
${nsd2_name}   slice_basic_middle_ns

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu
${password}   ${EMPTY}

${ns_id}   ${EMPTY}
${vnf_member_index}   middle
${vnf_ip_addr}   ${EMPTY}
${nst_config}   {netslice-vld: [ {name: slice_vld_mgmt, vim-network-name: %{VIM_MGMT_NET}} ] }


*** Test Cases ***
Create Slice VNF Descriptors
    [Documentation]   Onboards all the VNFDs required for the test: vnfd1_pkg and vnfd2_pkg (in the variables file)

    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd1_pkg}'
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd2_pkg}'


Create Slice NS Descriptors
    [Documentation]   Onboards all the NSDs required for the test: nsd1_pkg and nsd2_pkg (in the variables file)

    Create NSD   '%{PACKAGES_FOLDER}/${nsd1_pkg}'
    Create NSD   '%{PACKAGES_FOLDER}/${nsd2_pkg}'

Create Slice Templates
    [Documentation]   Onboards the Network Slice Templates: nst, nst2 (in the variables file)

    Create NST   '%{PACKAGES_FOLDER}/${nst}'
    Create NST   '%{PACKAGES_FOLDER}/${nst2}'

Network Slice First Instance
    [Documentation]   Instantiates the First NST recently onboarded (nst_name) and sets the instantiation id as a suite variable (nsi_id)
    ...               The slice contains 3 NS (1 shared)

    ${id}=   Create Network Slice   ${nst_name}   %{VIM_TARGET}   ${slice_name}   ${nst_config}   ${publickey}
    Set Suite Variable   ${nsi_id}   ${id}


Network Slice Second Instance
    [Documentation]   Instantiates the Second NST recently onboarded (nst2_name) and sets the instantiation id as a suite variable (nsi2_id)
    ...               The slice contains 2 NS (1 shared)

    ${id}=   Create Network Slice   ${nst2_name}   %{VIM_TARGET}   ${slice2_name}   ${nst_config}   ${publickey}
    Set Suite Variable   ${nsi2_id}   ${id}


First Network Slice Ns Count
    [Documentation]   Counts the NS in both slice instances and shoul be equal to 4

    ${slice1_count}=   Get Slice Ns Count   ${slice_name}
    ${slice2_count}=   Get Slice Ns Count   ${slice2_name}
    ${together}=   Evaluate   ${slice1_count} + ${slice2_count}
    Should Be Equal As Integers   ${together}   4


Get Middle Vnf Management Ip
    [Documentation]   Obtains the management IP of the shared NS main (only) VNF and sets it as a suite variable (mgmt_vnf_ip)

    ${middle_ns_id}=   Run and Return RC and Output   osm ns-list | grep ${middle_ns_name} | awk '{print $4}' 2>&1
    ${vnf_ip}   Get Vnf Management Ip Address   ${middle_ns_id}[1]   ${vnf_member_index}
    Run Keyword If   '${vnf_ip}' == '${EMPTY}'   Fatal Error   Variable \$\{ vnf_ip\} Empty
    Set Suite Variable   ${mgmt_vnf_ip}   ${vnf_ip}


Get First Slice Vnf IPs
    [Documentation]   Obtains the list of IPs addresses in the first slice and sets the list as a suite variable (slice1_vnfs_ips)

    # Get all the ns_id in the slice except the middle one
    @{ip_list}   Get Slice Vnf Ip Addresses   ${slice_name}
    Should Be True   ${ip_list} is not None
    Set Suite Variable   ${slice1_vnfs_ips}   ${ip_list}


Test Middle Ns Ping
    [Documentation]   Pings the slice middle vnf (mgmt_vnf_ip)

    Sleep   60s   Waiting for the network to be up
    # Ping to the middle VNF
    Test Connectivity   ${mgmt_vnf_ip}


Test Middle Vnf SSH Access
    [Documentation]   SSH access to the slice middle vnf (mgmt_vnf_ip) with the credentials provided in the variables file

    Sleep   30s   Waiting ssh daemon to be up
    Test SSH Connection   ${mgmt_vnf_ip}   ${username}   ${password}   ${privatekey}


Test First Slice Connectivity
    [Documentation]   SSH access to the slice middle vnf (mgmt_vnf_ip) with the credentials provided in the variables file
    ...               and pings all the IP addresses in the list (slice1_vnfs_ips)

    Ping Many   ${mgmt_vnf_ip}   ${username}   ${password}   ${privatekey}   @{slice1_vnfs_ips}


Delete Slice One Instance
    [Documentation]   Stops the slice instance (slice_name)

    [Tags]   cleanup

    Delete NSI   ${slice_name}


Second Network Slice Ns Count
    [Documentation]   Counts the NS in both slice instances and should be equal to 2

    ${slice1_count}=   Get Slice Ns Count   ${slice_name}
    ${slice2_count}=   Get Slice Ns Count   ${slice2_name}
    ${together}=   Evaluate   ${slice1_count} + ${slice2_count}
    Should Be Equal As Integers   ${together}   2

Get Second Slice Vnf IPs
    [Documentation]   Obtains the list of IPs addresses in the second slice and sets the list as a suite variable (slice2_vnfs_ips)

    # Get all the ns_id in the slice
    @{ip_list}   Get Slice Vnf Ip Addresses   ${slice2_name}
    Should Be True   ${ip_list} is not None
    Set Suite Variable   ${slice2_vnfs_ips}   ${ip_list}


Test Second Slice Connectivity
    [Documentation]   SSH access to the slice middle vnf (mgmt_vnf_ip) with the credentials provided in the variables file
    ...               and pings all the IP addresses in the list (slice2_vnfs_ips)

    Ping Many   ${mgmt_vnf_ip}   ${username}   ${password}   ${privatekey}   @{slice2_vnfs_ips}


Delete Slice Two Instance
    [Documentation]   Stops the slice instance (slice2_name)

    [Tags]   cleanup

    Delete NSI   ${slice2_name}


Delete Slice One Template
    [Documentation]   Deletes the NST (nst_name) from OSM

    [Tags]   cleanup

    Delete NST   ${nst_name}


Delete Slice Two Template
    [Documentation]   Deletes the NST (nst2_name) from OSM

    [Tags]   cleanup

    Delete NST   ${nst2_name}


Delete NS Descriptors
    [Documentation]   Deletes all the NSDs created for the test: nsd1_name, nsd2_name

    [Tags]   cleanup

    Delete NSD   ${nsd1_name}
    Delete NSD   ${nsd2_name}


Delete VNF Descriptors
    [Documentation]   Deletes all the VNFDs created for the test: vnfd1_name, vnfd2_name

    [Tags]   cleanup

    Delete VNFD   ${vnfd1_name}
    Delete VNFD   ${vnfd2_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptors, instance and templates

    Run Keyword If Any Tests Failed   Delete NSI   ${slice_name}
    Run Keyword If Any Tests Failed   Delete NSI   ${slice2_name}

    Run Keyword If Any Tests Failed   Delete NST   ${nst_name}
    Run Keyword If Any Tests Failed   Delete NST   ${nst2_name}

    Run Keyword If Any Tests Failed   Delete NSD   ${nsd1_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd2_name}

    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd1_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd2_name}


