#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [K8s-08] Simple K8s Scale.

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/k8scluster_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/juju_lib.robot

Force Tags   k8s_08   cluster_k8s   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# K8s cluster name and version
${k8scluster_name}   k8s08
${k8scluster_version}   v1

# NS and VNF descriptor package folder and ids
${vnfd_pkg}   charm-packages/native_k8s_scale_charm_vnf
${vnfd_name}   native_k8s_scale_charm-vnf
${nsd_pkg}   charm-packages/native_k8s_scale_charm_ns
${nsd_name}   native_k8s_scale_charm-ns

# NS instance name and configuration
${ns_name}   native-k8s-scale
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }
${ns_id}   ${EMPTY}
${publickey}   %{HOME}/.ssh/id_rsa.pub

# Username and SSH private key for accessing OSM host
${username}   ubuntu
${password}   ${EMPTY}
${privatekey}   %{OSM_RSA_FILE}

${model_name}   ${EMPTY}
${vnf_member_index}   native_k8s_scale_charm-vnf
${action_name}   changecontent
${kdu_name}   native-kdu
${application_name}   nginx
${scaling_group}   scale-kdu

*** Test Cases ***
Create Simple K8s Scale VNF Descriptor
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_pkg}'

Create Simple K8s Scale NS Descriptor
    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'

Add K8s Cluster To OSM
    Create K8s Cluster   %{K8S_CREDENTIALS}   ${k8scluster_version}   %{VIM_TARGET}   %{VIM_MGMT_NET}   ${k8scluster_name}

Network Service K8s Instance Test
    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}

Get KDU Model Name
    [Documentation]   Get the model name of the network service k8s instance

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${name}=   Get Model Name   %{OSM_HOSTNAME}   ${username}   ${password}   ${privatekey}   ${ns_id}   ${kdu_name}
    Set Suite Variable   ${model_name}   ${name}

Get Scale Count Before Scale Out
    [Documentation]   Get the scale count of the application of network service k8s instance

    Variable Should Exist   ${model_name}   msg=Model name is not available
    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${username}   ${password}   ${privatekey}   ${application_name}   ${model_name}
    Set Suite Variable   ${initial_kdu_count}   ${kdu_count}

Perform Manual KDU Scale Out
    [Documentation]   Scale out the application of network service k8s instance.

    ${ns_op_id_2}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_OUT

Check Scale Count After Scale Out
    [Documentation]   Check whether the scale count is more than one.

    Variable Should Exist   ${initial_kdu_count}   msg=Initial KDU count is not available
    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${username}   ${password}   ${privatekey}   ${application_name}   ${model_name}
    Run Keyword If   ${kdu_count} != ${initial_kdu_count} + 1   Fail   msg=There is no new KDU in the model after Scale Out

Perform Manual KDU Scale In
    [Documentation]   Scale in the application of network service k8s instance.

    ${ns_op_id_3}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_IN

Check Scale Count After Scale In
    [Documentation]   Check whether the scale count is one less.

    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${username}   ${password}   ${privatekey}   ${application_name}   ${model_name}
    Run Keyword If   ${kdu_count} != ${initial_kdu_count}   Fail   msg=There is the same number of KDU in the model after Scale In

Delete NS K8s Instance Test
    [Tags]   cleanup
    Delete NS   ${ns_name}

Remove K8s Cluster from OSM
    [Tags]   cleanup
    Delete K8s Cluster   ${k8scluster_name}

Delete NS Descriptor Test
    [Tags]   cleanup
    Delete NSD   ${nsd_name}

Delete VNF Descriptor Test
    [Tags]   cleanup
    Delete VNFD   ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim
    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}
    Run Keyword If Any Tests Failed   Delete K8s Cluster   ${k8scluster_name}
