#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [HEAL-04] Autohealing of NS

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnf_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/openstack_lib.robot

Force Tags   heal_04   cluster_heal   daily

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   autoheal_vnf
${vnfd_name}   autoheal_vnfd
${vdu_name}   autoheal_vnfd-VM
${vnf_index}   autoheal-basic-1
${nsd_pkg}   autoheal_ns
${nsd_name}   autoheal_nsd
# NS instance name and configuration
${ns_name}   heal_04
${ns_config}   {vld: [ {name: mgmt, vim-network-name: %{VIM_MGMT_NET}} ] }

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu
${password}   ${EMPTY}

${success_return_code}   0

#Healing wait time
${healing_pol_time}   15sec
${healing_max_wait_time}   10m

@{vim_vdus}   @{EMPTY}


*** Test Cases ***
Create VNF Descriptors
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_pkg}'


Create NS Descriptor
    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Network Service Instance Test
    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get NS Id
    [Tags]   cleanup
    ${variables}   Get Variables
    IF   not "\${ns_id}" in "${variables}"
        ${id}=   Get Ns Id   ${ns_name}
        Set Suite Variable   ${ns_id}   ${id}
    END


Get VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    log   ${vnf_id_list}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        log   ${vnf_id}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        Append To List   ${vim_vdus}   @{vdu_ids}
    END
    log Many   @{vim_vdus}


Get VNF Info
    Variable Should Exist   ${ns_id}   msg=NS is not available
    ${ip_addr}=   Get Vnf Management Ip Address   ${ns_id}   ${vnf_index}
    log   ${ip_addr}
    Set Suite Variable   ${vnf_ip_addr}   ${ip_addr}

    ${vnf_id}=   Get Vnf Id   ${ns_id}   ${vnf_index}
    Set Suite Variable   ${vnf_autoheal_id}   ${vnf_id}
    ${id}=   Get VNF VIM ID   ${vnf_id}
    Set Suite Variable   ${vdu_autoheal_id}   ${id}
    log   ${vdu_autoheal_id}

    @{autoheal_ip_list}=   Get Vnf Vdur IPs   ${vnf_autoheal_id}
    Set Suite Variable   @{autoheal_ip_list}   @{autoheal_ip_list}
    log   @{autoheal_ip_list}


Stop Autoheal VDU
    Variable Should Exist   ${vdu_autoheal_id}   msg=VDU is not available
    Halt Server   ${vdu_autoheal_id}
    Sleep   30


Wait For Autohealing To Be Completed
    ${healing_max_wait_time}=   Convert Time   ${healing_max_wait_time}   result_format=number
    ${healing_max_wait_time}=   Evaluate   ${healing_max_wait_time} * ${vim_timeout_multiplier}
    Wait Until Keyword Succeeds   ${healing_max_wait_time}   ${healing_pol_time}   Get Operations By Type   ${ns_id}   heal
    ${stdout}=   Get Operations By Type   ${ns_id}   heal
    Wait Until Keyword Succeeds   ${healing_max_wait_time}   ${healing_pol_time}   Check For NS Operation Ended   ${stdout}
    Check For NS Operation Completed   ${stdout}


Check VNF After Healing
    Variable Should Exist   ${vnf_autoheal_id}   msg=VNF is not available

    @{ip_list}=   Get Vnf Vdur IPs   ${vnf_autoheal_id}
    log   @{ip_list}
    Should Be Equal   ${ip_list}   ${autoheal_ip_list}   IP addresses have changed after healing

    ${id}=   Get VNF VIM ID   ${vnf_autoheal_id}
    log   ${id}
    Should Not Be Equal   ${id}   ${vdu_autoheal_id}   VDU id has not changed after healing


Update VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        FOR   ${id}   IN   @{vdu_ids}
            IF   not "${id}" in "@{vim_vdus}"
                Append To List   ${vim_vdus}   ${id}
            END
        END
    END
    log Many   @{vim_vdus}


Delete NS Instance
    [Tags]   cleanup
    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup
    Delete NSD   ${nsd_name}


Delete VNF Descriptors
    [Tags]   cleanup
    Delete VNFD   ${vnfd_name}


Delete Objects in VIM
    [Tags]   cleanup
    ${error}=   Set Variable   0
    FOR   ${vdu_id}   IN   @{vim_vdus}
        log   Checking if server ${vdu_id} is still in VIM
        ${status}=   Run Keyword And Ignore Error   Get Server Property   ${vdu_id}   id
        log   ${status}[0]
        IF   '${status}[0]' == 'PASS'
            ${error}=   Set Variable   1
            log   Deleting server ${vdu_id}
            Run Keyword And Ignore Error   Delete Server   ${vdu_id}
        END
    END
    IF   ${error}==1
        Fail   Some objects created by test were not deleted in VIM
    END


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}
