#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [HEAL-03] Healing of multiple VDUs

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnf_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/openstack_lib.robot

Force Tags   heal_03   cluster_heal   daily


Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_volumes_pkg}   several_volumes_vnf
${vnfd_volumes_name}   several_volumes-vnf
${vdu_volumes_name}   several_volumes-VM
${vnf_several_index}   several_volumes_vnf
${vnfd_charm_pkg}   charm-packages/native_manual_scale_charm_vnf
${vnfd_charm_name}   native_manual_scale_charm-vnf
${vdu_charm_name}   mgmtVM
${vnf_charm_index}   charm_vnf
${vnf_charm_scaling_group}   manual-scaling_mgmtVM
${vnf_charm_cloudinit_file}   /root/helloworld.txt
${vnf_charm_day1_file}   /home/ubuntu/first-touch
${nsd_pkg}   volumes_nativecharm_ns
${nsd_name}   volumes_nativecharm-ns

# NS instance name and configuration
${ns_name}   heal_03
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }
${scale_wait_time}   4min

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu
${password}   ${EMPTY}

${success_return_code}   0

@{vim_vdus}   @{EMPTY}
@{vim_volumes}   @{EMPTY}


*** Test Cases ***
Create VNF Descriptors
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_charm_pkg}'
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_volumes_pkg}'


Create NS Descriptor
    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Network Service Instance Test
    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get NS Id
    [Tags]   cleanup
    ${variables}   Get Variables
    IF   not "\${ns_id}" in "${variables}"
        ${id}=   Get Ns Id   ${ns_name}
        Set Suite Variable   ${ns_id}   ${id}
    END


Scale Out Charm VNF
    ${vnf_id}=   Get Vnf Id   ${ns_id}   ${vnf_charm_index}
    Set Suite Variable   ${vnf_charm_id}   ${vnf_id}
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_charm_id}
    ${vdurs}=   Get Length   ${vdur_list}
    Set Suite Variable   ${initial_vdur_count}   ${vdurs}
    Execute Manual VNF Scale   ${ns_name}   ${vnf_charm_index}   ${vnf_charm_scaling_group}   SCALE_OUT   ${scale_wait_time}
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_charm_id}
    ${vdurs}=   Get Length   ${vdur_list}
    Run Keyword If   ${vdurs} != ${initial_vdur_count} + 1   Fail   msg=There is no new VDU records in the VNF after Scale Out


Get VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    log   ${vnf_id_list}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        log   ${vnf_id}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        Append To List   ${vim_vdus}   @{vdu_ids}
    END
    FOR   ${vdu_id}   IN   @{vim_vdus}
        ${volumes_attached}=   Get Server Property   ${vdu_id}   volumes_attached
        ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
        IF   ${match} != @{EMPTY}
            IF   not "${match}[0]" in "@{vim_volumes}"
                Append To List   ${vim_volumes}   ${match}[0]
            END
        END
    END
    log Many   @{vim_vdus}
    log Many   @{vim_volumes}


Get Charm VNF Info
    Variable Should Exist   ${ns_id}   msg=NS is not available
    ${variables}   Get Variables
    IF   not "\${vnf_charm_id}" in "${variables}"
        ${vnf_id}=   Get Vnf Id   ${ns_id}   ${vnf_charm_index}
        Set Suite Variable   ${vnf_charm_id}   ${vnf_id}
    END
    ${id}=   Get VNF VIM ID   ${vnf_charm_id}
    @{vdu_charm_ids}=   Split String   ${id}
    Set Suite Variable   @{vdu_charm_ids}   @{vdu_charm_ids}
    log   ${vdu_charm_ids}[1]
    @{charm_ip_list}=   Get Vnf Vdur IPs   ${vnf_charm_id}
    Set Suite Variable   @{charm_ip_list}   @{charm_ip_list}


Get Volume VNF Info
    Variable Should Exist   ${ns_id}   msg=NS is not available
    ${ip_addr}=   Get Vnf Management Ip Address   ${ns_id}   ${vnf_several_index}
    log   ${ip_addr}
    Set Suite Variable   ${vnf_volumes_ip_addr}   ${ip_addr}

    ${vnf_id}=   Get Vnf Id   ${ns_id}   ${vnf_several_index}
    Set Suite Variable   ${vnf_volumes_id}   ${vnf_id}
    ${id}=   Get VNF VIM ID   ${vnf_id}
    Set Suite Variable   ${vdu_volumes_id}   ${id}
    log   ${vdu_volumes_id}

    @{volumes_ip_list}=   Get Vnf Vdur IPs   ${vnf_volumes_id}
    Set Suite Variable   @{volumes_ip_list}   @{volumes_ip_list}
    log   @{volumes_ip_list}


Stop VDUs
    Variable Should Exist   @{vdu_charm_ids}   msg=Charm VDUs are not available
    Variable Should Exist   ${vdu_volumes_id}   msg=Volume VDU is not available
    Stop Server   ${vdu_charm_ids}[1]
    Stop Server   ${vdu_charm_ids}[2]
    Stop Server   ${vdu_volumes_id}
    Sleep   30


Heal VDUs
    Variable Should Exist   ${vnf_charm_id}   msg=Charm VNF is not available
    Variable Should Exist   ${vnf_volumes_id}   msg=Volume VNF is not available
    Heal Network Service   ${ns_id}   --vnf ${vnf_volumes_id} --cause "Heal myvdu1 of several_volumes_vnf" --vdu several_volumes-VM --vnf ${vnf_charm_id} --cause "Heal two VMs of native_manual_scale_charm_vnf" --vdu mgmtVM --count-index 1 --run-day1 --vdu mgmtVM --count-index 2


Check VNFs After Healing
    Variable Should Exist   ${vnf_charm_id}   msg=Charm VNF is not available
    Variable Should Exist   ${vnf_volumes_id}   msg=Volume VNF is not available
    ${id}=   Get VNF VIM ID   ${vnf_charm_id}
    @{ids}=   Split String   ${id}
    Should Be Equal   ${vdu_charm_ids}[0]   ${ids}[0]   VDU[0] id has changed after healing
    Should Not Be Equal   ${vdu_charm_ids}[1]   ${ids}[1]   VDU[1] id has not changed after healing
    Should Not Be Equal   ${vdu_charm_ids}[2]   ${ids}[2]   VDU[2] id has not changed after healing

    ${id}=   Get VNF VIM ID   ${vnf_volumes_id}
    Should Not Be Equal   ${id}   ${vdu_volumes_id}   VDU id has not changed after healing

    ${ip}=   Get Vdu Attribute   ${vnf_charm_id}   ip-address   1
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${ip}   ${username}   ${password}   ${privatekey}   sudo ls ${vnf_charm_cloudinit_file}
    log   ${stdout}
    Check If remote File Exists   ${ip}   ${username}   ${password}   ${privatekey}   ${vnf_charm_day1_file}


Update VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vdu_updated}=   Create List
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        Append To List   ${vdu_updated}   @{vdu_ids}
        FOR   ${id}   IN   @{vdu_ids}
            IF   not "${id}" in "@{vim_vdus}"
                Append To List   ${vim_vdus}   ${id}
            END
        END
    END
    FOR   ${vdu_id}   IN   @{vdu_updated}
        ${volumes_attached}=   Get Server Property   ${vdu_id}   volumes_attached
        ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
        IF   ${match} != @{EMPTY}
            IF   not "${match}[0]" in "@{vim_volumes}"
                Append To List   ${vim_volumes}   ${match}[0]
            END
        END
    END
    log Many   @{vim_vdus}
    log Many   @{vim_volumes}


Delete NS Instance
    [Tags]   cleanup
    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup
    Delete NSD   ${nsd_name}


Delete VNF Descriptors
    [Tags]   cleanup
    Delete VNFD   ${vnfd_volumes_name}
    Delete VNFD   ${vnfd_charm_name}


Delete Objects in VIM
    [Tags]   cleanup
    ${error}=   Set Variable   0
    FOR   ${vol_id}   IN   @{vim_volumes}
        log   Checking if volume ${vol_id} is still in VIM
        ${exists}=   Check If Volume Exists   ${vol_id}
        IF   ${exists}!=0
            ${error}=   Set Variable   1
            log   Deleting volume ${vol_id}
            Run Keyword And Ignore Error   Delete Volume   ${vol_id}
        END
    END
    FOR   ${vdu_id}   IN   @{vim_vdus}
        log   Checking if server ${vdu_id} is still in VIM
        ${status}=   Run Keyword And Ignore Error   Get Server Property   ${vdu_id}   id
        log   ${status}[0]
        IF   '${status}[0]' == 'PASS'
            ${error}=   Set Variable   1
            log   Deleting server ${vdu_id}
            Run Keyword And Ignore Error   Delete Server   ${vdu_id}
        END
    END
    IF   ${error}==1
        Fail   Some objects created by test were not deleted in VIM
    END


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_volumes_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_charm_name}
