#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation     [FAIL-01] A single VNF with 2 VDUs, one of them unallocatable because of resources.

Library   OperatingSystem
Library   String
Library   Collections
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/openstack_lib.robot

Force Tags   fail_01   cluster_main   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   simple_2vm_vnf
${vnfd_name}   simple_2vm-vnf
${nsd_pkg}   simple_2vm_ns
${nsd_name}   simple_2vm-ns

# NS instance name and configuration
${ns_name_prefix}   fail_01
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }

# SSH keys to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub

# Default memory in GB for dataVM VDU (can be overwritten by VM_MEMORY_OVERRIDE environment variable)
# Only used if no quota is defined in Openstack's project (VDU tries to allocate the full memory of the quota)
${default_memory}   250

# NS launch timeout and polling time
${ns_launch_max_wait_time}   5min
${ns_launch_pol_time}   15

# Openstack delete operation timeout and polling time
${os_delete_max_wait_time}   6min
${os_delete_pol_time}   20


*** Test Cases ***
Create VNF Descriptor

    ${ram_quota}=   Get Project Quota   ram
    IF   ${ram_quota} == -1
        ${memory}=   Get Environment Variable    VM_MEMORY_OVERRIDE   default=${default_memory}
    ELSE
        ${memory}=   Evaluate   ${ram_quota} / 1000
    END
    log   ${memory}
    Create VNFD Overriding Fields   '%{PACKAGES_FOLDER}/${vnfd_pkg}'   virtual-compute-desc.1.virtual-memory.size=${memory}


Create NS Descriptor

    Create NSD  '%{PACKAGES_FOLDER}/${nsd_pkg}'


Cannot Instantiate Network Service

    ${rand}=  Generate Random String  4  [NUMBERS]
    ${ns_name}=  Catenate   SEPARATOR=_   ${ns_name_prefix}   ${rand}
    Set Suite Variable  ${ns_name}
    log   ${ns_name}
    ${id}=   Instantiate Network Service   ${ns_name}   ${nsd_name}   %{VIM_TARGET}   --config '${ns_config}' --ssh_keys ${publickey}
    log   ${id}
    Set Suite Variable  ${ns_id}  ${id}
    WAIT UNTIL KEYWORD SUCCEEDS   ${ns_launch_max_wait_time}   ${ns_launch_pol_time}   Check For Ns Instance To Configured   ${ns_name}
    ${rc}   ${stdout}=   run and return rc and output   osm ns-list --filter id="${ns_id}" | grep ${ns_id} | awk '{print $8}'
    log   ${stdout}
    Should Contain   ${stdout}   BROKEN
    ${rc}   ${stdout}=   run and return rc and output   osm ns-show ${ns_id} --literal | yq .errorDetail
    log   ${stdout}
    Should Contain   ${stdout}   Deploying at VIM: Error at create vdu


Delete NS Instance And Check VIM
    [Tags]   cleanup

    Delete NS  ${ns_name}
    WAIT UNTIL KEYWORD SUCCEEDS   ${os_delete_max_wait_time}   ${os_delete_pol_time}   Check NS Servers In VIM   ${ns_name}   0
 

Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD  ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD  ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]  Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed  Delete NS  ${ns_name}
    Run Keyword If Any Tests Failed  Delete NSD  ${nsd_name}
    Run Keyword If Any Tests Failed  Delete VNFD  ${vnfd_name}
