#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation     [BASIC-18] NS with a VLD with a virtual link profile.

Library   OperatingSystem
Library   String
Library   Collections
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot

Force Tags   basic_18   cluster_main   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   ubuntu_4ifaces_vnf
${vnfd_name}   ubuntu_4ifaces-vnf
${nsd_pkg}   ubuntu_4ifaces_ns
${nsd_name}   ubuntu_4ifaces-ns

# NS instance name and configuration
${ns_name}   basic_18
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu

# VNFs data
${vnf_member_index_1}   vnf1
${vnf_member_index_2}   vnf2
${iface1_name}   eth1
${iface2_name}   eth2
${iface3_name}   eth3
${datanet1_prefix}   ^192.168.10.*
${datanet2_prefix}   ^192.168.20.*
${datanet3_prefix}   ^192.168.30.*

${success_return_code}   0


*** Test Cases ***
Create VNF Descriptor

    Create VNFD  '%{PACKAGES_FOLDER}/${vnfd_pkg}'


Create NS Descriptor

    Create NSD  '%{PACKAGES_FOLDER}/${nsd_pkg}'


Instantiate Network Service

    ${id}=  Create Network Service  ${nsd_name}  %{VIM_TARGET}  ${ns_name}  ${ns_config}  ${publickey}
    Set Suite Variable  ${ns_id}  ${id}


Get Vnfs Info

    Variable Should Exist  ${ns_id}   msg=Network service instance is not available
    @{vnfr_list}=  Get Ns Vnfr Ids   ${ns_id}
    Log List   ${vnfr_list}
    Set Suite Variable   ${vnf_id1}   ${vnfr_list}[0]
    Set Suite Variable   ${vnf_id2}   ${vnfr_list}[1]
    ${ip}  Get Vnf Management Ip Address  ${ns_id}  ${vnf_member_index_1}
    Set Suite Variable  ${vnf1_ipmgmt}  ${ip}
    log  ${vnf1_ipmgmt}
    ${ip}  Get Vnf Management Ip Address  ${ns_id}  ${vnf_member_index_2}
    Set Suite Variable  ${vnf2_ipmgmt}  ${ip}
    log  ${vnf2_ipmgmt}


Check Vnf1 IPs

    Variable Should Exist  ${vnf_id1}  msg=VNF1 is not available
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id1} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface1_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet1_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet1_prefix}
    Set Suite Variable   ${vnf1_ip1}   ${stdout}
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id1} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface2_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet2_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet2_prefix}
    Set Suite Variable   ${vnf1_ip2}   ${stdout}
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id1} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface3_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet3_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet3_prefix}
    Set Suite Variable   ${vnf1_ip3}   ${stdout}



Check Vnf2 IPs

    Variable Should Exist  ${vnf_id1}  msg=VNF2 is not available
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id2} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface1_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet1_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet1_prefix}
    Set Suite Variable   ${vnf2_ip1}   ${stdout}
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id2} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface2_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet2_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet2_prefix}
    Set Suite Variable   ${vnf2_ip2}   ${stdout}
    ${rc}  ${stdout}=  Run and Return RC and Output  osm vnf-show ${vnf_id2} --literal | yq '.vdur[0].interfaces[] | select(.name == "${iface3_name}")' | yq '."ip-address"' | tr -d \\"
    Should Be Equal As Integers  ${rc}  ${success_return_code}  msg=${stdout}  values=False
    Should Match Regexp  ${stdout}  ${datanet3_prefix}  msg=${stdout} doesn't match subnet's regexp ${datanet3_prefix}
    Set Suite Variable   ${vnf2_ip3}   ${stdout}


Ping from Vnf1 to Vnf2

    Variable Should Exist  ${vnf1_ipmgmt}  msg=IP address of the data VNF '${vnf_member_index_1}' is not available
    Variable Should Exist  ${privatekey}  msg=SSH private key not available
    Sleep   30 seconds   Wait for SSH daemon to be up
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf1_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ip addr ; ping -c 5 ${vnf2_ipmgmt}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf1_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ping -c 5 ${vnf2_ip1}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf1_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ping -c 5 ${vnf2_ip2}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf1_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ping -c 5 ${vnf2_ip3}
    log  ${stdout}


Ping from Vnf2 to Vnf1

    Variable Should Exist  ${vnf1_ipmgmt}  msg=IP address of the data VNF '${vnf_member_index_2}' is not available
    Variable Should Exist  ${privatekey}  msg=SSH private key not available
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf2_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ip addr ; ping -c 5 ${vnf1_ipmgmt}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf2_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ip addr ; ping -c 5 ${vnf1_ip1}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf2_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ip addr ; ping -c 5 ${vnf1_ip2}
    log  ${stdout}
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf2_ipmgmt}   ${username}   ${EMPTY}   ${privatekey}   ip addr ; ping -c 5 ${vnf1_ip3}
    log  ${stdout}


Delete NS Instance
    [Tags]   cleanup

    Delete NS  ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD  ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD  ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]  Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed  Delete NS  ${ns_name}
    Run Keyword If Any Tests Failed  Delete NSD  ${nsd_name}
    Run Keyword If Any Tests Failed  Delete VNFD  ${vnfd_name}
