#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [SA-07] Events or alarms coming from SA-related VNFs in the NS.

Library   OperatingSystem
Library   String
Library   Collections
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/prometheus_lib.robot

Variables   %{ROBOT_DEVOPS_FOLDER}/resources/sa_07-alarms_from_sa-related_vnfs_data.py

Force Tags   sa_07   cluster_sa   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS instantiation parameters
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }

# NS and VNF descriptor package folder and ids
${vnfd_pkg}   cirros_alarm_vnf
${vnfd_name}   cirros_alarm-vnf
${vnfd_file}   cirros_alarm_vnfd.yaml
${new_vnfd_pkg}   new_cirros_alarm_vnf
${nsd_pkg}   cirros_alarm_ns
${nsd_name}   cirros_alarm-ns

# NS instance name and id
${ns_id}   ${EMPTY}
${ns_name}   sa_07-alarm_test

# Webhook NS and VNF descriptor package folder and ids
${ws_vnfd_pkg}   hackfest_basic_vnf
${ws_nsd_pkg}   hackfest_basic_ns
${ws_vnfd_name}   hackfest_basic-vnf
${ws_nsd_name}   hackfest_basic-ns

# Webhook NS instance name and id
${ws_ns_id}   ${EMPTY}
${ws_ns_name}   sa_07-webhook_service_test

# Webhook NS Variables
${ws_vnf_member_index}   vnf
${ws_vnf_ip_addr}   ${EMPTY}
${ws_log_file}   webhook.log
${ws_port}   5212
${ws_alarm_msg}   notify_alarm

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu
${password}   ${EMPTY}

# Prometheus polling interval and retries
${prometheus_poll_retries}   15 times
${prometheus_poll_timeout}   1 minute

# Prometheus metrics to retrieve
${metric_name}   osm_cpu_utilization

${success_return_code}   0


*** Test Cases ***
Create Webhook Service VNF Descriptor

    Create VNFD   '%{PACKAGES_FOLDER}/${ws_vnfd_pkg}'


Create Webhook Service NS Descriptor

    Create NSD   '%{PACKAGES_FOLDER}/${ws_nsd_pkg}'


Instantiate Webhook Service Network Service

    ${id}=   Create Network Service   ${ws_nsd_name}   %{VIM_TARGET}   ${ws_ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ws_ns_id}   ${id}


Get Webhook Service VNF IP Address

    ${ip_addr}=   Get Vnf Management Ip Address   ${ws_ns_id}   ${ws_vnf_member_index}
    Log   ${ip_addr}
    Set Suite Variable   ${ws_vnf_ip_addr}   ${ip_addr}


Start Webhook Service

    Variable Should Exist   ${privatekey}   msg=SSH private key not available
    Sleep   40 seconds   Wait for SSH daemon to be up
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${ws_vnf_ip_addr}   ${username}   ${password}   ${privatekey}   while true; do { echo -e 'HTTP/1.0 200 OK\r\nContent-Length: 0\r\n\r\n'; test; } | nc -l '${ws_port}'; done > '${ws_log_file}' 2>&1 &


Create VNF Descriptor

    ${rc}   ${stdout}=   Run and Return RC and Output   mkdir '%{PACKAGES_FOLDER}/${new_vnfd_pkg}' && WEBHOOK_URL="http://${ws_vnf_ip_addr}:${ws_port}" envsubst < '%{PACKAGES_FOLDER}/${vnfd_pkg}'/'${vnfd_file}' > '%{PACKAGES_FOLDER}/${new_vnfd_pkg}'/'${vnfd_file}'
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Create VNFD   '%{PACKAGES_FOLDER}/${new_vnfd_pkg}'


Create NS Descriptor

    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Instantiate Network Service

    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get Alarm Metric

    Variable Should Exist   ${prometheus_poll_retries}   msg=Metric polling retries is not available
    Variable Should Exist   ${prometheus_poll_timeout}   msg=Metric polling timeout is not available
    Variable Should Exist   ${prometheus_host}   msg=Prometheus address is not available
    Variable Should Exist   ${prometheus_port}   msg=Prometheus port is not available
    Variable Should Exist   ${metric_name}   msg=Prometheus metric name is not available
    ${metric_filter}=   Set Variable   ns_id=${ns_id}
    ${metric_value}=   Wait Until Keyword Succeeds   ${prometheus_poll_retries}   ${prometheus_poll_timeout}   Get Metric   ${prometheus_host}   ${prometheus_port}   ${prometheus_user}   ${prometheus_password}   ${metric_name}   ${metric_filter}
    Run Keyword If   ${metric_value} <= 0   Fail   msg=The metric '${metric_name}' value is '${metric_value}'


Check Alarms Were Received

    Wait Until Keyword Succeeds   6 times   40 seconds   Execute Remote Command Check Rc Return Output   ${ws_vnf_ip_addr}   ${username}   ${password}   ${privatekey}   cat '${ws_log_file}' | grep '${ws_alarm_msg}' | grep '${ns_name}'


Delete NS Instance
    [Tags]   cleanup

    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD   ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD   ${vnfd_name}


Delete Webhook Service NS Instance
    [Tags]   cleanup

    Delete NS   ${ws_ns_name}


Delete Webhook Service NS Descriptor
    [Tags]   cleanup

    Delete NSD   ${ws_nsd_name}


Delete Webhook Service VNF Descriptor
    [Tags]   cleanup

    Delete VNFD   ${ws_vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}
    Run Keyword If Any Tests Failed   Delete NS   ${ws_ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${ws_nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${ws_vnfd_name}
    Delete Temporary Descriptor Folder   '%{PACKAGES_FOLDER}/${new_vnfd_pkg}'


Delete Temporary Descriptor Folder
    [Documentation]   Removes the temporary package folder created for the test
    [Arguments]   ${folder_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   rm -rf '${folder_name}'
    Log   ${stdout}


