#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [BASIC-20] Manual VDU Scaling to test juju applications are scaling for native charms (testing Bug-1609).

Library   OperatingSystem
Library   String
Library   Collections

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot

Force Tags   basic_20   cluster_main   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   charm-packages/native_manual_scale_charm_vnf
${vnfd_name}   native_manual_scale_charm-vnf
${nsd_pkg}   charm-packages/native_manual_scale_charm_ns
${nsd_name}   native_manual_scale_charm-ns

# NS instance name and configuration
${ns_name}   basic_20_manual_vdu_scaling_test
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }
${vnf_member_index}   vnf

# SSH public key file
${publickey}   %{HOME}/.ssh/id_rsa.pub

# Scaling group name of VDU instances
${scaling_group}   manual-scaling_mgmtVM

${scale_wait_time}   4min

*** Test Cases ***
Create VNF Descriptor

    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_pkg}'

Create NS Descriptor

    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Instantiate Network Service

    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get Vnf Id

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vnfr_list}=   Get Ns Vnfr Ids   ${ns_id}
    Log List   ${vnfr_list}
    Set Suite Variable   ${vnf_id}   ${vnfr_list}[0]


Get Vdus Before Scale Out
    [Documentation]   Get the number of VDU records before the manual scaling.

    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Set Suite Variable   ${initial_vdur_count}   ${vdurs}


Get Application Name Before Scale Out
    [Documentation]   Get the application names.

    @{name_list}=   Get Application Names   ${ns_name}
    Log List   ${name_list}
    Set Suite Variable   ${application_list}   ${name_list}


Check Number of Duplicated Application Name Before Scale Out
    [Documentation]   Check the application names if there are 2 application name in the application list.

    ${count_duplicated}=   Get Length   ${application_list}
    Run Keyword If   ${count_duplicated} != 2   Fail   msg=There are not 2 application name in the application list before Scale Out


Perform Manual Vdu Scale Out

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_OUT   ${scale_wait_time}


Check Vdus After Scale Out
    [Documentation]   Check whether there is one more VDU after scaling or not.

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Run Keyword If   ${vdurs} != ${initial_vdur_count} + 1   Fail   msg=There is no new VDU records in the VNF after Scale Out


Get Application Name After Scale Out
    [Documentation]   Get the application names.

    @{name_list}=   Get Application Names   ${ns_name}
    Log List   ${name_list}
    Set Suite Variable   ${application_list}   ${name_list}


Check Number of Duplicated Application Name After Scale Out
    [Documentation]   Check the application names if there are 3 application name in the application list.

    ${count_duplicated}=   Get Length   ${application_list}
    Run Keyword If   ${count_duplicated} != 3   Fail   msg=There are not 2 application name in the application list after Scale Out


Perform Manual Vdu Scale In

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_IN


Check Vdus After Scaling In
    [Documentation]   Check whether there is one less VDU after scaling or not.

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Run Keyword If   ${vdurs} != ${initial_vdur_count}   Fail   msg=There is the same number of VDU records in the VNF after Scale In


Get Application Name After Scale In
    [Documentation]   Get the application names.

    @{name_list}=   Get Application Names   ${ns_name}
    Log List   ${name_list}
    Set Suite Variable   ${application_list}   ${name_list}


Check Number of Duplicated Application Name After Scale In
    [Documentation]   Check the application names if there are 2 application name in the application list.

    ${count_duplicated}=   Get Length   ${application_list}
    Run Keyword If   ${count_duplicated} != 2   Fail   msg=There are not 2 application name in the application list after Scale In


Delete NS Instance
    [Tags]   cleanup

    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD   ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD   ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}

    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}

    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}

