#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   Library to deploy and delete NS, and run operations on them.

Library   DateTime


*** Variables ***
${success_return_code}   0
${ns_launch_pol_time}   30sec
${ns_delete_pol_time}   15sec
${ns_action_max_wait_time}   1min
${ns_action_pol_time}   15sec
${vnf_scale_pol_time}   15sec
${healing_pol_time}   15sec
${vim_timeout_multiplier}   %{OSM_VIM_MULTIPLIER_TIMEOUT=1.0}


*** Keywords ***
Create Network Service
    [Arguments]   ${nsd}   ${vim_name}   ${ns_name}   ${ns_config}   ${publickey}   ${ns_launch_max_wait_time}=5min   ${config_file}=${EMPTY}

    ${ns_launch_max_wait_time}=   Convert Time   ${ns_launch_max_wait_time}   result_format=number
    ${ns_launch_max_wait_time}=   Evaluate   ${ns_launch_max_wait_time} * ${vim_timeout_multiplier}
    Log   ${ns_launch_max_wait_time}
    ${config_attr}   Set Variable If   '${ns_config}'!='${EMPTY}'   --config '${ns_config}'   \
    ${sshkeys_attr}   Set Variable If   '${publickey}'!='${EMPTY}'   --ssh_keys ${publickey}   \
    ${config_file_attr}   Set Variable If   '${config_file}'!='${EMPTY}'   --config_file '${config_file}'   \

    ${ns_id}=   Instantiate Network Service   ${ns_name}   ${nsd}   ${vim_name}   ${config_attr} ${sshkeys_attr} ${config_file_attr}
    Log   ${ns_id}

    WAIT UNTIL KEYWORD SUCCEEDS   ${ns_launch_max_wait_time}   ${ns_launch_pol_time}   Check For NS Instance To Configured   ${ns_name}
    Check For NS Instance For Failure   ${ns_name}
    [Return]   ${ns_id}


Instantiate Network Service
    [Arguments]   ${ns_name}   ${nsd}   ${vim_name}   ${ns_extra_args}

    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-create --ns_name ${ns_name} --nsd_name ${nsd} --vim_account ${vim_name} ${ns_extra_args}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Update Network Service
    [Arguments]   ${ns_id}   ${update_type}   ${ns_update_config}   ${ns_update_timeout}

    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-update ${ns_id} --updatetype ${update_type} --config ${ns_update_config} --timeout ${ns_update_timeout} --wait
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Vnf Management Ip Address
    [Arguments]   ${ns_id}   ${vnf_member_index}

    Should Not Be Empty   ${ns_id}
    Should Not Be Empty   ${vnf_member_index}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-list --filter member-vnf-index-ref=${vnf_member_index} | grep ${ns_id} | awk '{print $14}' 2>&1
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Vnf Id
    [Arguments]   ${ns_id}   ${vnf_member_index}

    Should Not Be Empty   ${ns_id}
    Should Not Be Empty   ${vnf_member_index}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-list --filter member-vnf-index-ref=${vnf_member_index} | grep ${ns_id} | awk '{print $2}' 2>&1
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Ns Id
    [Arguments]   ${ns_name}

    Should Not Be Empty   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-list | grep ${ns_name} | awk '{print $4}' 2>&1
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Not Be Empty   ${stdout}
    [Return]   ${stdout}


Get Ns List
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-list 2>&1
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Ns Vnf List
    [Arguments]   ${ns_id}

    Should Not Be Empty   ${ns_id}
    @{vnf_list_string}=   Run and Return RC and Output   osm vnf-list | grep ${ns_id} | awk '{print $2}' 2>&1
    # Returns a String of vnf_id and needs to be converted into a list
    @{vnf_list} =   Split String   ${vnf_list_string}[1]
    Log List   ${vnf_list}
    [Return]   @{vnf_list}


Get Ns Ip List
    [Arguments]   @{vnf_list}

    should not be empty   @{vnf_list}
    @{temp_list}=   Create List
    FOR   ${vnf_id}   IN   @{vnf_list}
        Log   ${vnf_id}
        @{vnf_ip_list}   Get Vnf Ip List   ${vnf_id}
        @{temp_list}=   Combine Lists   ${temp_list}   ${vnf_ip_list}
    END
    should not be empty   ${temp_list}
    [return]   @{temp_list}


Get Vnf Ip List
    [arguments]   ${vnf_id}

    should not be empty   ${vnf_id}
    @{vnf_ip_list_string}=   run and return rc and output   osm vnf-list --filter id=${vnf_id} | grep -o '[0-9]\\{1,3\\}\\.[0-9]\\{1,3\\}\\.[0-9]\\{1,3\\}\\.[0-9]\\{1,3\\}' | sort -t: -u -k1,1 2>&1
    # returns a string of ip addresses and needs to be converted into a list
    should not be empty   ${vnf_ip_list_string}[1]
    @{vnf_ip_list} =   split string   ${vnf_ip_list_string}[1]
    Log Many   ${vnf_ip_list}
    should not be empty   ${vnf_ip_list}
    [return]   @{vnf_ip_list}


Check For Ns Instance To Configured
    [arguments]   ${ns_name}

    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server list
    Log   ${stdout}
    ${rc}   ${stdout}=   run and return rc and output   osm ns-list --filter name="${ns_name}"
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Contain Any   ${stdout}   READY   BROKEN

Check For NS Instance For Failure
    [Arguments]   ${ns_name}

    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server list
    Log   ${stdout}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-list --filter name="${ns_name}"
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Not Contain   ${stdout}   BROKEN

Check For NS Instance To Be Deleted
    [Arguments]   ${ns}

    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server list
    Log   ${stdout}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-list | awk '{print $2}' | grep ${ns}
    Should Not Be Equal As Strings   ${stdout}   ${ns}

Delete NS
    [Documentation]   Delete ns
    [Arguments]   ${ns}   ${ns_delete_max_wait_time}=4min

    ${ns_delete_max_wait_time}=   Convert Time   ${ns_delete_max_wait_time}   result_format=number
    ${ns_delete_max_wait_time}=   Evaluate   ${ns_delete_max_wait_time} * ${vim_timeout_multiplier}
    Log   ${ns_delete_max_wait_time}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-delete ${ns}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}

    WAIT UNTIL KEYWORD SUCCEEDS   ${ns_delete_max_wait_time}   ${ns_delete_pol_time}   Check For NS Instance To Be Deleted   ${ns}

Execute NS Action
    [Documentation]   Execute an action over the desired NS.
    ...               Parameters are given to this function in key=value format (one argument per key/value pair).
    ...               Return the ID of the operation associated to the executed action.
    ...               Examples of execution:
    ...                   \${ns_op_id}=   Execute NS Action   \${ns_name}   \${ns_action}   \${vnf_member_index}
    ...                   \${ns_op_id}=   Execute NS Action   \${ns_name}   \${ns_action}   \${vnf_member_index}   \${param1}=\${value1}   \${param2}=\${value2}

    [Arguments]   ${ns_name}   ${ns_action}   ${vnf_member_index}   @{action_params}

    ${params}=   Set Variable   ${EMPTY}
    FOR   ${param}   IN   @{action_params}
        ${match}   ${param_name}   ${param_value} =   Should Match Regexp   ${param}   (.+)=(.+)   msg=Syntax error in parameters
        ${params}=   Catenate   SEPARATOR=   ${params}   "${param_name}":"${param_value}",
    END
    ${osm_ns_action_command}=   Set Variable   osm ns-action --action_name ${ns_action} --vnf_name ${vnf_member_index}
    ${osm_ns_action_command}=   Run Keyword If   '${params}'!='${EMPTY}'   Catenate   ${osm_ns_action_command}   --params '{${params}}'
    ...   ELSE   Set Variable   ${osm_ns_action_command}
    ${osm_ns_action_command}=   Catenate   ${osm_ns_action_command}   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   ${osm_ns_action_command}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Wait Until Keyword Succeeds   ${ns_action_max_wait_time}   ${ns_action_pol_time}   Check For NS Operation Ended   ${stdout}
    Check For NS Operation Completed   ${stdout}
    [Return]   ${stdout}


Execute NS K8s Action
    [Documentation]   Execute an action over the desired K8s NS.
    ...               Parameters are given to this function in key=value format (one argument per key/value pair).
    ...               Return the ID of the operation associated to the executed action.
    ...               Examples of execution:
    ...                   \${ns_op_id}=   Execute NS Action   \${ns_name}   \${ns_action}   \${vnf_member_index}
    ...                   \${ns_op_id}=   Execute NS Action   \${ns_name}   \${ns_action}   \${vnf_member_index}   \${param1}=\${value1}   \${param2}=\${value2}

    [Arguments]   ${ns_name}   ${ns_action}   ${vnf_member_index}   ${kdu_name}   @{action_params}

    ${params}=   Set Variable   ${EMPTY}
    FOR   ${param}   IN   @{action_params}
        ${match}   ${param_name}   ${param_value} =   Should Match Regexp   ${param}   (.+)=(.+)   msg=Syntax error in parameters
        ${params}=   Catenate   SEPARATOR=   ${params}   "${param_name}":"${param_value}",
    END
    ${osm_ns_action_command}=   Set Variable   osm ns-action --action_name ${ns_action} --vnf_name ${vnf_member_index} --kdu_name ${kdu_name}
    ${osm_ns_action_command}=   Run Keyword If   '${params}'!='${EMPTY}'   Catenate   ${osm_ns_action_command}   --params '{${params}}'
    ...   ELSE   Set Variable   ${osm_ns_action_command}
    ${osm_ns_action_command}=   Catenate   ${osm_ns_action_command}   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   ${osm_ns_action_command}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Wait Until Keyword Succeeds   ${ns_action_max_wait_time}   ${ns_action_pol_time}   Check For NS Operation Ended   ${stdout}
    Check For NS Operation Completed   ${stdout}
    [Return]   ${stdout}


Execute Manual VNF Scale
    [Documentation]   Execute a manual VNF Scale action.
    ...               The parameter 'scale_type' must be SCALE_IN or SCALE_OUT.
    ...               Return the ID of the operation associated to the executed scale action.

    [Arguments]   ${ns_name}   ${vnf_member_index}   ${scaling_group}   ${scale_type}   ${vnf_scale_max_wait_time}=2min

    ${vnf_scale_max_wait_time}=   Convert Time   ${vnf_scale_max_wait_time}   result_format=number
    ${vnf_scale_max_wait_time}=   Evaluate   ${vnf_scale_max_wait_time} * ${vim_timeout_multiplier}
    Should Contain Any   ${scale_type}   SCALE_IN   SCALE_OUT   msg=Unknown scale type: ${scale_type}   values=False
    ${osm_vnf_scale_command}=   Set Variable   osm vnf-scale --scaling-group ${scaling_group}
    ${osm_vnf_scale_command}=   Run Keyword If   '${scale_type}'=='SCALE_IN'   Catenate   ${osm_vnf_scale_command}   --scale-in
    ...   ELSE   Catenate   ${osm_vnf_scale_command}   --scale-out
    ${osm_vnf_scale_command}=   Catenate   ${osm_vnf_scale_command}   ${ns_name}   ${vnf_member_index}
    ${rc}   ${stdout}=   Run and Return RC and Output   ${osm_vnf_scale_command}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Wait Until Keyword Succeeds   ${vnf_scale_max_wait_time}   ${vnf_scale_pol_time}   Check For NS Operation Ended   ${stdout}
    Check For NS Operation Completed   ${stdout}
    [Return]   ${stdout}


Heal Network Service
    [Documentation]   Execute healing operation over one NS.
    ...               Return the ID of the operation associated to the executed healing action.

    [Arguments]   ${ns_name}   ${params}   ${healing_max_wait_time}=10m

    Should Not Be Empty   ${ns_name}
    Should Not Be Empty   ${params}
    ${healing_max_wait_time}=   Convert Time   ${healing_max_wait_time}   result_format=number
    ${healing_max_wait_time}=   Evaluate   ${healing_max_wait_time} * ${vim_timeout_multiplier}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-heal ${ns_name} ${params}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Wait Until Keyword Succeeds   ${healing_max_wait_time}   ${healing_pol_time}   Check For NS Operation Ended   ${stdout}
    Check For NS Operation Completed   ${stdout}
    [Return]   ${stdout}


Get Operations List
    [Arguments]   ${ns_name}

    Should Not Be Empty   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-op-list ${ns_name}
    Log   ${stdout}
    Log   ${rc}
    Should Be Equal As Integers   ${rc}   ${success_return_code}


Check For NS Operation Completed
    [Documentation]   Check wheter the status of the desired operation is "COMPLETED" or not.

    [Arguments]   ${ns_operation_id}

    Should Not Be Empty   ${ns_operation_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-op-show ${ns_operation_id} --literal | yq .operationState | tr -d \\"
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Contain   ${stdout}   COMPLETED   msg=The ns-action with id ${ns_operation_id} was not completed   values=False


Check For NS Operation Failed
    [Documentation]   Check wheter the status of the desired operation is "FAILED" or not.

    [Arguments]   ${ns_operation_id}

    Should Not Be Empty   ${ns_operation_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-op-show ${ns_operation_id} --literal | yq .operationState | tr -d \\"
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Contain   ${stdout}   FAILED   msg=Timeout waiting for ns-action with id ${ns_operation_id}   values=False


Check For NS Operation Ended
    [Documentation]   Check wheter the status of the desired operation is "FAILED" or "COMPLETED".

    [Arguments]   ${ns_operation_id}

    Should Not Be Empty   ${ns_operation_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-op-show ${ns_operation_id} --literal | yq .operationState | tr -d \\"
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Contain Any   ${stdout}   FAILED   COMPLETED   msg=Timeout waiting for ns-action with id ${ns_operation_id}   values=False


Get Ns Vnfr Ids
    [Documentation]   Return a list with the IDs of the VNF records of a NS instance.

    [Arguments]   ${ns_id}

    Should Not Be Empty   ${ns_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-list | grep ${ns_id} | awk '{print $2}' 2>&1
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    @{vdur} =   Split String   ${stdout}
    [Return]   @{vdur}


Get Vnf Vdur Names
    [Documentation]   Return a list with the names of the VDU records of a VNF instance.

    [Arguments]   ${vnf_id}

    Should Not Be Empty   ${vnf_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --literal | yq .vdur[].name | tr -d \\"
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    @{vdur} =   Split String   ${stdout}
    [Return]   @{vdur}


Get Vnf Kdu Replica Count
    [Documentation]   Return the number of KDU replicas (empty if none) of a VNF instance.

    [Arguments]   ${vnf_id}   ${kdu_name}

    Should Not Be Empty   ${vnf_id}
    Should Not Be Empty   ${kdu_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --kdu ${kdu_name} | yq .config.replicaCount | tr -d \\"
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    ${return} =   Set Variable If   '${stdout}' == 'null'   ${EMPTY}   ${stdout}
    [Return]   ${return}


Get Application Names
    [Documentation]   Return the list of the application of a VNF instance.

    [Arguments]   ${ns_name}

    Should Not Be Empty   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-show ${ns_name} --literal | yq ._admin.deployed.VCA[].application | tr -d \\"
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    @{app_names} =   Split String   ${stdout}
    [Return]   ${app_names}


Get VDU Affinity Group Name
    [Documentation]   Return the affinity group name for a NF

    [Arguments]   ${ns_name}   ${vnf_id}   ${vdur_id}

    ${rc}   ${affinity_group_id}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --literal | yq '.vdur[] | select(.id == "'${vdur_id}'")' | yq '."affinity-or-anti-affinity-group-id"[0]' | tr -d \\" 2>&1
    Log   ${affinity_group_id}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${affinity_group_id}   values=False
    ${rc}   ${affinity_group_name}=   Run and Return RC and Output   osm ns-show ${ns_name} --literal | yq '."affinity-or-anti-affinity-group"[] | select(.id == "'${affinity_group_id}'")' | yq '.name' | tr -d \\" 2>&1
    Log   ${affinity_group_name}
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${affinity_group_name}   values=False
    [Return]   ${affinity_group_name}

Get Operations By Type
    [Documentation]   Keyword to get the operation by type

    [Arguments]   ${ns_id}   ${type}

    Should Not Be Empty   ${ns_id}
    Should Not Be Empty   ${type}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm ns-op-list ${ns_id} | grep ${type} | awk '{print $2}' 2>&1
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Not Be Empty   ${stdout}
    [Return]   ${stdout}
