#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [K8s-13] Two Helm-based KDU stored in public and private OCI repositories.

Library   OperatingSystem
Library   String
Library   Collections
Library   Process

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/connectivity_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/k8scluster_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/repo_lib.robot

Force Tags   k8s_13   cluster_k8s   daily   azure

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   two_helm_oci_knf
${vnfd_name}   two_helm_oci_knf
${nsd_pkg}   two_helm_oci_ns
${nsd_name}   two_helm_oci_ns

# NS instance name and configuration
${ns_name}   ldap
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}}], additionalParamsForVnf: [ {member-vnf-index: two_helm_oci, additionalParamsForKdu: [ {kdu_name: haproxy, additionalParams: {service: {type: LoadBalancer }, adminPassword: admin}}, {kdu_name: ldap, additionalParams: {service: {type: LoadBalancer }, adminPassword: admin}} ] } ] }

${ns_id}   ${EMPTY}
${publickey}   ${EMPTY}
${vnf_member_index}   two_helm_oci
${kdu1_name}   haproxy
${kdu2_name}   ldap
${upgrade_action}   upgrade
${rollback_action}   rollback
${replica_count}   3

# OCI helm repo configuration
${repo_name}   osm-gitlab
${repo_uri}   %{OCI_REGISTRY_URL}
${repo_user}   %{OCI_REGISTRY_USER}
${repo_password}   %{OCI_REGISTRY_PASSWORD}

*** Test Cases ***
Create Package For CNF
    [Documentation]   Create Package For CNF
    [Tags]   prepare
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_pkg}'

Create Package For NS
    [Documentation]   Create Package For NS
    [Tags]   prepare
    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'

Create Helm OCI Repo
    [Documentation]   Create Helm OCI Repo for openldap kdu
    [Tags]   prepare
    Create Repo   ${repo_name}   ${repo_uri}   helm-chart   ${repo_user}   ${repo_password}   --oci

Create Network Service Instance
    [Documentation]   Create Network Service Instance
    [Tags]   prepare
    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}

Get Ns Id
    [Documentation]   Get ID of NS instance
    [Tags]   verify
    ${id}=   Get Ns Id   ${ns_name}
    Set Suite Variable   ${ns_id}   ${id}

Get Vnf Id
    [Documentation]   Get ID of VNF
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vnfr_list}=   Get Ns Vnfr Ids   ${ns_id}
    Log List   ${vnfr_list}
    Set Suite Variable   ${vnf_id}   ${vnfr_list}[0]

Execute Upgrade Operation over first KDU
    [Documentation]   Execute Upgrade Operation over first KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute NS K8s Action   ${ns_name}   ${upgrade_action}   ${vnf_member_index}   ${kdu1_name}   replicaCount=${replica_count}

Check Replicas After Upgrade Operation over first KDU
    [Documentation]   Check Replicas After Upgrade Operation over first KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${count}=   Get Vnf Kdu Replica Count   ${vnf_id}   ${kdu1_name}
    Log   ${count}
    Should Be Equal As Integers   ${count}   ${replica_count}

Execute Rollback Operation over first KDU
    [Documentation]   Execute Rollback Operation over first KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute NS K8s Action   ${ns_name}   ${rollback_action}   ${vnf_member_index}   ${kdu1_name}

Check Replicas After Rollback Operation over first KDU
    [Documentation]   Check Replicas After Rollback Operation over first KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${count}=   Get Vnf Kdu Replica Count   ${vnf_id}   ${kdu1_name}
    Log   ${count}
    Should Be Empty   ${count}

Execute Upgrade Operation over second KDU
    [Documentation]   Execute Upgrade Operation over second KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute NS K8s Action   ${ns_name}   ${upgrade_action}   ${vnf_member_index}   ${kdu2_name}   replicaCount=${replica_count}

Check Replicas After Upgrade Operation over second KDU
    [Documentation]   Check Replicas After Upgrade Operation over second KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${count}=   Get Vnf Kdu Replica Count   ${vnf_id}   ${kdu2_name}
    Log   ${count}
    Should Be Equal As Integers   ${count}   ${replica_count}

Execute Rollback Operation over second KDU
    [Documentation]   Execute Rollback Operation over second KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute NS K8s Action   ${ns_name}   ${rollback_action}   ${vnf_member_index}   ${kdu2_name}

Check Replicas After Rollback Operation over second KDU
    [Documentation]   Check Replicas After Rollback Operation over second KDU
    [Tags]   verify
    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${count}=   Get Vnf Kdu Replica Count   ${vnf_id}   ${kdu2_name}
    Log   ${count}
    Should Be Empty   ${count}

Delete Network Service Instance
    [Documentation]   Delete Network Service Instance
    [Tags]   cleanup
    Delete NS   ${ns_name}

Delete Helm OCI Repo
    [Documentation]   Delete Helm OCI Repo
    [Tags]   prepare
    Delete Repo   ${repo_name}

Delete NS Descriptor Test
    [Documentation]   Delete NS Descriptor Test
    [Tags]   cleanup
    Delete NSD   ${nsd_name}

Delete VNF Descriptor Test
    [Documentation]   Delete VNF Descriptor Test
    [Tags]   cleanup
    Delete VNFD   ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim
    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}
