#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [HEAL-01] Healing of a multi-volume VDU

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnf_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/openstack_lib.robot

Force Tags   heal_01   cluster_heal   daily


Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_volumes_pkg}   several_volumes_vnf
${vnfd_volumes_name}   several_volumes-vnf
${vdu_volumes_name}   several_volumes-VM
${vnf_several_index}   several_volumes_vnf
${vnfd_charm_pkg}   charm-packages/native_manual_scale_charm_vnf
${vnfd_charm_name}   native_manual_scale_charm-vnf
${nsd_pkg}   volumes_nativecharm_ns
${nsd_name}   volumes_nativecharm-ns

# NS instance name and configuration
${ns_name}   heal_01
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }
${ns_timeout}   6min

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${privatekey}   %{HOME}/.ssh/id_rsa
${username}   ubuntu
${password}   ${EMPTY}

${success_return_code}   0

@{vim_vdus}   @{EMPTY}
@{vim_volumes}   @{EMPTY}


*** Test Cases ***
Create VNF Descriptors
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_charm_pkg}'
    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_volumes_pkg}'


Create NS Descriptor
    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Network Service Instance Test
    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}   ${ns_timeout}
    Set Suite Variable   ${ns_id}   ${id}


Get NS Id
    [Tags]   cleanup
    ${variables}   Get Variables
    IF   not "\${ns_id}" in "${variables}"
        ${id}=   Get Ns Id   ${ns_name}
        Set Suite Variable   ${ns_id}   ${id}
    END


Get VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    Log   ${vnf_id_list}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        Log   ${vnf_id}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        Append To List   ${vim_vdus}   @{vdu_ids}
    END
    FOR   ${vdu_id}   IN   @{vim_vdus}
        ${volumes_attached}=   Get Server Property   ${vdu_id}   volumes_attached
        ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
        IF   ${match} != @{EMPTY}
            IF   not "${match}[0]" in "@{vim_volumes}"
                Append To List   ${vim_volumes}   ${match}[0]
            END
        END
    END
    Log Many   @{vim_vdus}
    Log Many   @{vim_volumes}


Get Volume VNF Info
    Variable Should Exist   ${ns_id}   msg=NS is not available
    ${ip_addr}=   Get Vnf Management Ip Address   ${ns_id}   ${vnf_several_index}
    Log   ${ip_addr}
    Set Suite Variable   ${vnf_volumes_ip_addr}   ${ip_addr}

    ${vnf_id}=   Get Vnf Id   ${ns_id}   ${vnf_several_index}
    Set Suite Variable   ${vnf_volumes_id}   ${vnf_id}
    ${id}=   Get VNF VIM ID   ${vnf_id}
    Set Suite Variable   ${vdu_volumes_id}   ${id}
    Log   ${vdu_volumes_id}

    @{volumes_ip_list}=   Get Vnf Vdur IPs   ${vnf_volumes_id}
    Set Suite Variable   @{volumes_ip_list}   @{volumes_ip_list}
    Log   @{volumes_ip_list}


Get Volumes Info
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnfpkg-show ${vnfd_volumes_name} --literal | yq '.vdu[0]."virtual-storage-desc" | length'
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    ${num_virtual_storage}=   Convert To Integer   ${stdout}
    Set Suite Variable   ${vnf_num_volumes}   ${num_virtual_storage}
    Log   ${vnf_num_volumes}
    ${volumes_attached}=   Get Server Property   ${vdu_volumes_id}   volumes_attached
    ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
    Set Suite Variable   ${volume_id}   ${match}[0]


Check VDU Disks
    Variable Should Exist   ${vnf_volumes_ip_addr}   msg=VNF is not available
    Sleep   20 seconds   Wait for SSH daemon to be up
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf_volumes_ip_addr}   ${username}   ${password}   ${privatekey}   sudo lsblk -l
    Log   ${stdout}
    ${lines}=   Get Lines Containing String   ${stdout}   disk
    ${num_lines}=   Get Line Count   ${lines}
    Run Keyword If   ${num_lines} < ${vnf_num_volumes}   Fail   msg=Number of disks (${num_lines}) is less than specified in VDU (${vnf_num_volumes})


Delete Persistent Volume VDU
    Variable Should Exist   ${vdu_volumes_id}   msg=VDU is not available
    Delete Server   ${vdu_volumes_id}
    Sleep   20


Heal Persistent Volume VDU
    Variable Should Exist   ${vnf_volumes_id}   msg=VNF is not available
    Heal Network Service   ${ns_id}   --vnf ${vnf_volumes_id} --cause "Heal VM of volumes_vnf" --vdu ${vdu_volumes_name}


Check VNF After Healing
    Variable Should Exist   ${vnf_volumes_id}   msg=VNF is not available

    @{ip_list}=   Get Vnf Vdur IPs   ${vnf_volumes_id}
    Log   @{ip_list}
    Should Be Equal   ${ip_list}   ${volumes_ip_list}   IP addresses have changed after healing

    ${id}=   Get VNF VIM ID   ${vnf_volumes_id}
    Log   ${id}
    Should Not Be Equal   ${id}   ${vdu_volumes_id}   VDU id has not changed after healing

    ${volumes_attached}=   Get Server Property   ${id}   volumes_attached
    ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
    Should Be Equal   ${match}[0]   ${volume_id}   Volume id has changed after healing

    Sleep   30 seconds   Wait for SSH daemon to be up
    ${stdout}=   Execute Remote Command Check Rc Return Output   ${vnf_volumes_ip_addr}   ${username}   ${password}   ${privatekey}   sudo lsblk -l
    Log   ${stdout}
    ${lines}=   Get Lines Containing String   ${stdout}   disk
    ${num_lines}=   Get Line Count   ${lines}
    Run Keyword If   ${num_lines} < ${vnf_num_volumes}   Fail   msg=Number of disks (${num_lines}) is less than specified in VDU (${vnf_num_volumes})


Update VIM Objects
    Variable Should Exist   ${ns_id}   msg=NS is not available
    @{vdu_updated}=   Create List
    @{vnf_id_list}=   Get Ns Vnf List   ${ns_id}
    FOR   ${vnf_id}   IN   @{vnf_id_list}
        ${id}=   Get VNF VIM ID   ${vnf_id}
        @{vdu_ids}=   Split String   ${id}
        Append To List   ${vdu_updated}   @{vdu_ids}
        FOR   ${id}   IN   @{vdu_ids}
            IF   not "${id}" in "@{vim_vdus}"
                Append To List   ${vim_vdus}   ${id}
            END
        END
    END
    FOR   ${vdu_id}   IN   @{vdu_updated}
        ${volumes_attached}=   Get Server Property   ${vdu_id}   volumes_attached
        ${match}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
        IF   ${match} != @{EMPTY}
            IF   not "${match}[0]" in "@{vim_volumes}"
                Append To List   ${vim_volumes}   ${match}[0]
            END
        END
    END
    Log Many   @{vim_vdus}
    Log Many   @{vim_volumes}


Delete NS Instance
    [Tags]   cleanup
    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup
    Delete NSD   ${nsd_name}


Delete VNF Descriptors
    [Tags]   cleanup
    Delete VNFD   ${vnfd_volumes_name}
    Delete VNFD   ${vnfd_charm_name}


Delete Objects in VIM
    [Tags]   cleanup
    ${error}=   Set Variable   0
    FOR   ${vol_id}   IN   @{vim_volumes}
        Log   Checking if volume ${vol_id} is still in VIM
        ${exists}=   Check If Volume Exists   ${vol_id}
        IF   ${exists}!=0
            ${error}=   Set Variable   1
            Log   Deleting volume ${vol_id}
            Run Keyword And Ignore Error   Delete Volume   ${vol_id}
        END
    END
    FOR   ${vdu_id}   IN   @{vim_vdus}
        Log   Checking if server ${vdu_id} is still in VIM
        ${status}=   Run Keyword And Ignore Error   Get Server Property   ${vdu_id}   id
        Log   ${status}[0]
        IF   '${status}[0]' == 'PASS'
            ${error}=   Set Variable   1
            Log   Deleting server ${vdu_id}
            Run Keyword And Ignore Error   Delete Server   ${vdu_id}
        END
    END
    IF   ${error}==1
        Fail   Some objects created by test were not deleted in VIM
    END


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_volumes_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_charm_name}
    Run Keyword If Any Tests Failed   Delete Objects in VIM
