#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [BASIC-29] NS with a single VNF and two VDU linked by a VLD with ipv6-profile.

Library   OperatingSystem
Library   String
Library   Collections
Library   SSHLibrary
Library   JSONLibrary

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnf_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/openstack_lib.robot

Force Tags   basic_29   cluster_main   daily

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   ipv6profile_2vm_vnf
${vnfd_name}   ipv6profile_2vm-vnf
${nsd_pkg}   ipv6profile_2vm_ns
${nsd_name}   ipv6profile_2vm-ns

# NS instance name and configuration
${ns_name}   basic_29
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }

# SSH keys and username to be used
${publickey}   %{HOME}/.ssh/id_rsa.pub
${username}   ubuntu

# VNFs data
${vnf_member_index_1}   vnf
${internal_prefix}   ^2001:db8::*

${success_return_code}   0


*** Test Cases ***
Create VNF Descriptor

    Create VNFD   '%{PACKAGES_FOLDER}/${vnfd_pkg}'


Create NS Descriptor

    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Instantiate Network Service

    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get Vnf Info

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vnfr_list}=   Get Ns Vnfr Ids   ${ns_id}
    Log List   ${vnfr_list}
    Set Suite Variable   ${vnf_id}   ${vnfr_list}[0]
    ${ip}   Get Vnf Management Ip Address   ${ns_id}   ${vnf_member_index_1}
    Set Suite Variable   ${vnf_ipmgmt}   ${ip}
    Log   ${vnf_ipmgmt}


Check Vnf IPs

    Variable Should Exist   ${vnf_id}   msg=VNF is not available
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --literal | yq '.vdur[0].interfaces[] | select(."vnf-vld-id" == "internal")' | yq -r '."ip-address"'
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Should Match Regexp   ${stdout}   ${internal_prefix}   msg=${stdout} doesn't match subnet's regexp ${internal_prefix}
    ${rc}   ${stdout}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --literal | yq '.vdur[1].interfaces[] | select(."vnf-vld-id" == "internal")' | yq -r '."ip-address"'
    Should Be Equal As Integers   ${rc}   ${success_return_code}   msg=${stdout}   values=False
    Should Match Regexp   ${stdout}   ${internal_prefix}   msg=${stdout} doesn't match subnet's regexp ${internal_prefix}

Check that ipv6 address_mode and ra_mode are set

    ${rc}   ${vim_info}=   Run and Return RC and Output   osm vnf-show ${vnf_id} --literal | yq '.vdur[] | select(."count-index" == 0)' | yq -r '.vim_info[].interfaces[].vim_info'
    ${subnet_id}=   Get Regexp Matches   ${vim_info}   {ip_address: '2001:db8::.*', subnet_id: ([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})}   1
    ${subnet_info}=   Get Subnet   ${subnet_id}[0]
    ${json_object}=   Convert String to JSON   ${subnet_info}
    ${address_mode}=   Get From Dictionary   ${json_object}   ipv6_address_mode
    ${ra_mode}=   Get From Dictionary   ${json_object}   ipv6_ra_mode
    Should Be Equal   ${address_mode}   dhcpv6-stateful   msg=ipv6_address_mode does not equals to dhcpv6-stateful
    Should Be Equal   ${ra_mode}   dhcpv6-stateful   msg=ipv6_ra_mode does not equals to dhcpv6-stateful


Delete NS Instance
    [Tags]   cleanup

    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD   ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD   ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}
