#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Variables ***
${success_return_code}   0


*** Keywords ***
Get Server Flavor ID
    [Arguments]   ${server_id}

    Should Not Be Empty   ${server_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server show ${server_id} | grep flavor | awk 'NR>1{print $1}' RS='(' FS=')'
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Flavor Properties
    [Arguments]   ${flavor_id}

    Should Not Be Empty   ${flavor_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack flavor show ${flavor_id} | grep properties | awk -F '|' '{print $3}'
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Check NS Servers In VIM
    [Documentation]   Check if the number of servers in Openstack by filtering by NS name is as expected.
    [Arguments]   ${ns_name}   ${number}

    Should Not Be Empty   ${ns_name}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server list | grep ${ns_name} | awk '{print $4}'
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    @{servers} =   Split String   ${stdout}
    ${n_servers}=   Get Length   ${servers}
    Should Be Equal As Integers   ${number}   ${n_servers}


Check If Volume Bootable
    [Documentation]   Check the volume properties and return whether it is bootable or not.
    [Arguments]   ${volume_id}

    Should Not Be Empty   ${volume_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack volume show ${volume_id} | grep bootable | awk '{print $4}'
    log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Project Quota
    [Documentation]   Get a field (ram, cores, ports, etc.) from quota in current project.
    [Arguments]   ${field}

    Should Not Be Empty   ${field}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack quota show -f yaml | yq '.[] | select(.Resource=="${field}") | .Limit'
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Server Property
    [Documentation]   Get a field (flavor, image, volumes_attached, etc.) from a server.
    [Arguments]   ${server_id}   ${field}

    Should Not Be Empty   ${server_id}
    Should Not Be Empty   ${field}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server show ${server_id} -c ${field} -f value
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Check VM In Server Group
    [Documentation]   Check if a given server is included in an Openstack server group
    [Arguments]   ${server_id}   ${server_group_name}

    Should Not Be Empty   ${server_group_name}
    Should Not Be Empty   ${server_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server group show ${server_group_name} -c members -f value
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    Should Contain   ${stdout}   ${server_id}


Set Volume Id
    [Documentation]   Sets the root volume ID and ordinary volume ID by checking bootable property of volumes
    [Arguments]   ${volume_match}   ${index}

    Should Not Be Empty   ${volume_match}   ${index}
    ${bootable}=   Check If Volume Bootable   ${volume_match}[${index}]
    Set Suite Variable   ${bootable}
    IF   '${bootable}'=='true'
        Set Suite Variable   ${root_vol_id}   ${volume_match}[${index}]
        IF   ${index}==0
            Set Suite Variable   ${ordinary_vol_id}   ${volume_match}[1]
        ELSE
            Set Suite Variable   ${ordinary_vol_id}   ${volume_match}[0]
        END
    END


Get Persistent Volumes Attached To Vm
    [Documentation]   Sets the root volume ID and ordinary volume ID by checking bootable property of volumes
    [Arguments]   ${ns_id}

    Variable Should Exist   ${ns_id}   msg=NS is not available
    ${vnfs_list}=   Get Ns Vnf List   ${ns_id}
    ${vim_id}=   Get VNF VIM ID   ${vnfs_list}[0]
    ${volumes_attached}=   Get Server Property   ${vim_id}   volumes_attached
    log   ${volumes_attached}
    ${stdout}=   Get Regexp Matches   ${volumes_attached}   '([0-9a-f\-]+)'   1
    log   ${stdout}
    [Return]   ${stdout}


Check If Volume Exists
    [Documentation]   Checks if a volume id exists
    [Arguments]   ${volume_id}

    Should Not Be Empty   ${volume_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack volume list | grep ${volume_id} | wc -l
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Delete Volume
    [Documentation]   Delete a volume by its identifier
    [Arguments]   ${volume_id}

    Should Not Be Empty   ${volume_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack volume delete ${volume_id}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Get Subnet
    [Documentation]   Get subnet information in JSON format
    [Arguments]   ${subnet_id}

    Should Not Be Empty   ${subnet_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack subnet show ${subnet_id} -f json
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Stop Server
    [Documentation]   Stop a server
    [Arguments]   ${server_id}

    Should Not Be Empty   ${server_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server stop ${server_id}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Halt Server
    [Documentation]   Reset a server
    [Arguments]   ${server_id}

    Should Not Be Empty   ${server_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server set --state error ${server_id}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Delete Server
    [Documentation]   Delete a server
    [Arguments]   ${server_id}

    Should Not Be Empty   ${server_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack server delete ${server_id}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}


Create Flavor
    [Documentation]   Create a Flavor
    [Arguments]   ${flavor_name}   ${cpu}=1   ${ram}=1024   ${disk}=10

    Should Not Be Empty   ${flavor_name}
    ${rc}   ${flavor_id}=   Run and Return RC and Output   openstack flavor create ${flavor_name} --ram ${ram} --disk ${disk} --vcpus ${cpu} -f yaml | yq '.id' | tr -d \\"
    Log   ${flavor_id}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${flavor_id}


Delete Flavor
    [Documentation]   Delete a Flavor
    [Arguments]   ${flavor_id}

    Should Not Be Empty   ${flavor_id}
    ${rc}   ${stdout}=   Run and Return RC and Output   openstack flavor delete ${flavor_id}
    Log   ${stdout}
    Should Be Equal As Integers   ${rc}   ${success_return_code}
    [Return]   ${stdout}
