#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.

*** Settings ***
Documentation   [BASIC-16] Advanced onboarding with override and complex scaling (3 initial instances, scaled by two).

Library   OperatingSystem
Library   String
Library   Collections

Resource   %{ROBOT_DEVOPS_FOLDER}/lib/vnfd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/nsd_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ns_lib.robot
Resource   %{ROBOT_DEVOPS_FOLDER}/lib/ssh_lib.robot

Force Tags   basic_16   cluster_main   daily   regression

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${vnfd_pkg}   hackfest_basic_metrics_vnf
${vnfd_name}   hackfest_basic_metrics-vnf
${nsd_pkg}   hackfest_basic_metrics_ns
${nsd_name}   hackfest_basic-ns-metrics

# NS instance name and configuration
${ns_name}   basic_16
${ns_config}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ] }

# SSH public keys file
${publickey}   %{HOME}/.ssh/id_rsa.pub

# Initial, delta, min and max number of VDU instances
${init_instances}   3
${min_instances}   3
${max_instances}   5
${delta_instances}   2
${scaling_group}   vdu_autoscale
${vnf_member_index}   vnf


*** Test Cases ***
Create Scaling VNF Descriptor

    Create VNFD Overriding Fields   '%{PACKAGES_FOLDER}/${vnfd_pkg}'   df.0.instantiation-level.0.vdu-level.0.number-of-instances=${init_instances};df.0.vdu-profile.0.min-number-of-instances=${min_instances};df.0.vdu-profile.0.max-number-of-instances=${max_instances};df.0.scaling-aspect.0.aspect-delta-details.deltas.0.vdu-delta.0.number-of-instances=${delta_instances}


Create Scaling NS Descriptor

    Create NSD   '%{PACKAGES_FOLDER}/${nsd_pkg}'


Instantiate Scaling Network Service

    ${id}=   Create Network Service   ${nsd_name}   %{VIM_TARGET}   ${ns_name}   ${ns_config}   ${publickey}
    Set Suite Variable   ${ns_id}   ${id}


Get Vnf Id

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vnfr_list}=   Get Ns Vnfr Ids   ${ns_id}
    Log List   ${vnfr_list}
    Set Suite Variable   ${vnf_id}   ${vnfr_list}[0]


Check Vdus Before Scale Out
    [Documentation]   Check the number of VDUs instances before the manual scaling.

    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Set Suite Variable   ${initial_vdur_count}   ${vdurs}
    Run Keyword If   ${init_instances} != ${initial_vdur_count}   Fail   msg=Instantiated VDUs (${initial_vdur_count}) don't match specified number in descriptor (${init_instances})


Perform Manual Vdu Scale Out

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_OUT


Check Vdus After Scale Out
    [Documentation]   Check whether there is one more VDU after scaling or not.

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Run Keyword If   ${vdurs} != ${initial_vdur_count} + ${delta_instances}   Fail   msg=VDU instances in the VNF (${vdurs}) don't match after Scale Out (${initial_vdur_count} + ${delta_instances})


Perform Manual Vdu Scale In

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    ${ns_op_id}=   Execute Manual VNF Scale   ${ns_name}   ${vnf_member_index}   ${scaling_group}   SCALE_IN


Check Vdus After Scaling In
    [Documentation]   Check whether there is one less VDU after scaling or not.

    Variable Should Exist   ${ns_id}   msg=Network service instance is not available
    @{vdur_list}=   Get Vnf Vdur Names   ${vnf_id}
    Log List   ${vdur_list}
    ${vdurs}=   Get Length   ${vdur_list}
    Run Keyword If   ${vdurs} != ${initial_vdur_count}   Fail   msg=There is the same number of VDU records in the VNF after Scale In


Delete NS Instance
    [Tags]   cleanup

    Delete NS   ${ns_name}


Delete NS Descriptor
    [Tags]   cleanup

    Delete NSD   ${nsd_name}


Delete VNF Descriptor
    [Tags]   cleanup

    Delete VNFD   ${vnfd_name}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suite Cleanup: Deleting descriptors and NS instance

    Run Keyword If Any Tests Failed   Delete NS   ${ns_name}
    Run Keyword If Any Tests Failed   Delete NSD   ${nsd_name}
    Run Keyword If Any Tests Failed   Delete VNFD   ${vnfd_name}

