*** Comments ***
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.


*** Settings ***
Documentation   [GITOPS-01] Advanced Cluster Management.

Library   OperatingSystem
Library   String
Library   Collections
Library   Process

Resource   ../lib/advanced_cluster_mgmt_lib.resource

Test Tags   gitops_01   cluster_gitops

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# OKA packages
${OKA_FOLDER}   '%{PACKAGES_FOLDER}/oka'
${OKA_PKG1}   apps/testapp
${OKA_NAME1}   testapp
${OKA_PROFILE1}   app-profile
${OKA_PKG2}   apps/testacme
${OKA_NAME2}   testacme
${OKA_PROFILE2}   app-profile
${OKA_PKG3}   apps/jenkins
${OKA_NAME3}   jenkins
${OKA_PROFILE3}   app-profile

# Cluster
${CLUSTER_NAME}   gitops01
${CLUSTER_DESCRIPTION}   "Cluster for gitops01 Robot test suite"
${CLUSTER_VM_SIZE}   Standard_D2_v2
${CLUSTER_NODES}   2
${REGION_NAME}   northeurope
${RESOURCE_GROUP}   OSM-CTIO
${KUBERNETES_VERSION}   "1.30"

# KSUs
${KSU_NAME1}   testapp
${KSU_NAME2}   testacme
${KSU_NAME3}   jenkins


*** Test Cases ***
Add OKA Packages For The Testsuite
    [Documentation]   Add OKA packages for the testsuite.
    [Tags]   oka   create
    Add OKA Package   ${OKA_NAME1}   '${OKA_FOLDER}/${OKA_PKG1}'   ${OKA_PROFILE1}
    Add OKA Package   ${OKA_NAME2}   '${OKA_FOLDER}/${OKA_PKG2}'   ${OKA_PROFILE2}
    Add OKA Package   ${OKA_NAME3}   '${OKA_FOLDER}/${OKA_PKG3}'   ${OKA_PROFILE3}

Create Cluster For Testsuite in Azure
    [Documentation]   Create K8s cluster for the testsuite.
    [Tags]   cluster   create
    ${id}=   Create Cluster   ${CLUSTER_NAME}   %{VIM_TARGET}   ${CLUSTER_DESCRIPTION}   ${CLUSTER_VM_SIZE}   ${KUBERNETES_VERSION}   ${CLUSTER_NODES}   ${REGION_NAME}   ${RESOURCE_GROUP}
    Set Suite Variable   ${CLUSTER_ID}   ${id}
    Log   ${CLUSTER_ID}

Get Cluster Kubeconfig
    [Documentation]   Get Kubeconfig for the cluster.
    [Tags]   cluster
    ${rc}   ${stdout}=   Run And Return Rc And Output
    ...    osm cluster-show ${CLUSTER_NAME} -o jsonpath='{.credentials}' | yq -y
    Should Be Equal As Integers   ${rc}   ${SUCCESS_RETURN_CODE}
    Log   ${stdout}

Create KSU3
    [Documentation]   Create KSU3 based on ${OKA_NAME3} with params.
    [Tags]   ksu   create
    ${id}=   Create KSU   ${KSU_NAME3}   ${KSU_NAME3}   ${CLUSTER_NAME}   app-profile   ${OKA_NAME3}   ${OKA_FOLDER}/apps/jenkins-params.yaml
    Set Suite Variable   ${KSU_ID3}   ${id}
    Log   ${KSU_ID3}

Delete KSU3
    [Documentation]   Delete KSU3 based on ${OKA_NAME3}.
    [Tags]   cleanup   ksu
    Delete KSU   ${KSU_ID3}

Delete Cluster For Testsuite
    [Documentation]   Delete K8s cluster for the testsuite.
    [Tags]   cleanup   cluster
    Delete Cluster   ${CLUSTER_NAME}

Delete OKA Packages For The Testsuite
    [Documentation]   Delete OKA packages for the testsuite.
    [Tags]   cleanup   oka
    Delete OKA Package   ${OKA_NAME1}
    Delete OKA Package   ${OKA_NAME2}
    Delete OKA Package   ${OKA_NAME3}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting cluster, KSUs, OKAs
    Run Keyword If Any Tests Failed   Delete Cluster   ${CLUSTER_NAME}
    Run Keyword If Any Tests Failed   Delete OKA Package   ${OKA_NAME1}
    Run Keyword If Any Tests Failed   Delete OKA Package   ${OKA_NAME2}
    Run Keyword If Any Tests Failed   Delete OKA Package   ${OKA_NAME3}
