*** Comments ***
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.


*** Settings ***
Documentation   [K8s-08] Simple K8s Scale.

Library   OperatingSystem
Library   String
Library   Collections
Library   Process
Library   SSHLibrary

Resource   ../lib/vnfd_lib.resource
Resource   ../lib/nsd_lib.resource
Resource   ../lib/ns_lib.resource
Resource   ../lib/ns_operation_lib.resource
Resource   ../lib/ssh_lib.resource
Resource   ../lib/k8scluster_lib.resource
Resource   ../lib/juju_lib.resource

Test Tags   k8s_08   cluster_k8s   regression   azure

Suite Teardown   Run Keyword And Ignore Error   Suite Cleanup


*** Variables ***
# NS and VNF descriptor package folder and ids
${VNFD_PKG}   charm-packages/native_k8s_scale_charm_vnf
${VNFD_NAME}   native_k8s_scale_charm-vnf
${NSD_PKG}   charm-packages/native_k8s_scale_charm_ns
${NSD_NAME}   native_k8s_scale_charm-ns

${KDU_NAME}   native-kdu
${APPLICATION_NAME}   nginx
${VNF_MEMBER_INDEX}   native_k8s_scale_charm-vnf
${INSTANTIATION_COUNT}   2

# NS instance name and configuration
${NS_NAME}   native-k8s-scale
${NS_CONFIG}   {vld: [ {name: mgmtnet, vim-network-name: %{VIM_MGMT_NET}} ], additionalParamsForVnf: [{member-vnf-index: ${VNF_MEMBER_INDEX}, additionalParamsForKdu: [{ kdu_name: ${KDU_NAME}, additionalParams: { overlay: { applications: { ${APPLICATION_NAME}: { scale: ${INSTANTIATION_COUNT} } }}}}]}]}
${NS_ID}   ${EMPTY}
${PUBLICKEY}   %{HOME}/.ssh/id_rsa.pub

# Username and SSH private key for accessing OSM host
${USERNAME}   ubuntu
${PASSWORD}   ${EMPTY}
${PRIVATEKEY}   %{OSM_RSA_FILE}

${MODEL_NAME}   ${EMPTY}
${ACTION_NAME}   changecontent
${SCALING_GROUP}   scale-kdu


*** Test Cases ***
Create Simple K8s Scale VNF Descriptor
    [Documentation]   Upload VNF package for the testsuite.
    Create VNFD   '%{PACKAGES_FOLDER}/${VNFD_PKG}'

Create Simple K8s Scale NS Descriptor
    [Documentation]   Upload NS package for the testsuite.
    Create NSD   '%{PACKAGES_FOLDER}/${NSD_PKG}'

Network Service K8s Instance Test
    [Documentation]   Instantiate NS for the testsuite.
    ${id}=   Create Network Service   ${NSD_NAME}   %{VIM_TARGET}   ${NS_NAME}   ${NS_CONFIG}   ${PUBLICKEY}
    Set Suite Variable   ${NS_ID}   ${id}

Get KDU Model Name
    [Documentation]   Get the model name of the network service k8s instance
    Variable Should Exist   ${NS_ID}   msg=Network service instance is not available
    ${name}=   Get Model Name   %{OSM_HOSTNAME}   ${USERNAME}   ${PASSWORD}   ${PRIVATEKEY}   ${NS_ID}   ${KDU_NAME}
    Set Suite Variable   ${MODEL_NAME}   ${name}

Get Scale Count Before Scale Out
    [Documentation]   Get the scale count of the application of network service k8s instance
    Variable Should Exist   ${MODEL_NAME}   msg=Model name is not available
    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${USERNAME}   ${PASSWORD}   ${PRIVATEKEY}   ${APPLICATION_NAME}   ${MODEL_NAME}
    Set Suite Variable   ${INITIAL_KDU_COUNT}   ${kdu_count}
    IF   ${INITIAL_KDU_COUNT} != ${INSTANTIATION_COUNT}   Fail   msg=Instantiation parameters were not applied.

Perform Manual KDU Scale Out
    [Documentation]   Scale out the application of network service k8s instance.
    ${ns_op_id_2}=   Execute Manual VNF Scale   ${NS_NAME}   ${VNF_MEMBER_INDEX}   ${SCALING_GROUP}   SCALE_OUT
    Log   ${ns_op_id_2}

Check Scale Count After Scale Out
    [Documentation]   Check whether the scale count is more than one.

    Variable Should Exist   ${INITIAL_KDU_COUNT}   msg=Initial KDU count is not available
    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${USERNAME}   ${PASSWORD}   ${PRIVATEKEY}   ${APPLICATION_NAME}   ${MODEL_NAME}
    IF   ${kdu_count} != ${INITIAL_KDU_COUNT} + 1   Fail   msg=There is no new KDU in the model after Scale Out

Perform Manual KDU Scale In
    [Documentation]   Scale in the application of network service k8s instance.
    ${ns_op_id_3}=   Execute Manual VNF Scale   ${NS_NAME}   ${VNF_MEMBER_INDEX}   ${SCALING_GROUP}   SCALE_IN
    Log   ${ns_op_id_3}

Check Scale Count After Scale In
    [Documentation]   Check whether the scale count is one less.
    ${kdu_count}=   Get Scale Number   %{OSM_HOSTNAME}   ${USERNAME}   ${PASSWORD}   ${PRIVATEKEY}   ${APPLICATION_NAME}   ${MODEL_NAME}
    IF   ${kdu_count} != ${INITIAL_KDU_COUNT}   Fail   msg=There is the same number of KDU in the model after Scale In

Delete NS K8s Instance Test
    [Documentation]   Delete NS instance.
    [Tags]   cleanup
    Delete NS   ${NS_NAME}

Delete NS Descriptor Test
    [Documentation]   Delete NS package from OSM.
    [Tags]   cleanup
    Delete NSD   ${NSD_NAME}

Delete VNF Descriptor Test
    [Documentation]   Delete NF package from OSM.
    [Tags]   cleanup
    Delete VNFD   ${VNFD_NAME}


*** Keywords ***
Suite Cleanup
    [Documentation]   Test Suit Cleanup: Deleting Descriptor, instance and vim
    Run Keyword If Any Tests Failed   Delete NS   ${NS_NAME}
    Run Keyword If Any Tests Failed   Delete NSD   ${NSD_NAME}
    Run Keyword If Any Tests Failed   Delete VNFD   ${VNFD_NAME}
