# Copyright ETSI Contributors and Others.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import click
from osmclient.cli_commands import utils
from prettytable import PrettyTable
import logging

logger = logging.getLogger("osmclient")


@click.command(
    name="package-create", short_help="Create empty VNF or NS package structure"
)
@click.argument("package-type")
@click.argument("package-name")
@click.option(
    "--base-directory",
    default=".",
    help=('(NS/VNF/NST) Set the location for package creation. Default: "."'),
)
@click.option(
    "--image",
    default="image-name",
    help='(VNF) Set the name of the vdu image. Default "image-name"',
)
@click.option(
    "--vdus", default=1, help="(VNF) Set the number of vdus in a VNF. Default 1"
)
@click.option(
    "--vcpu", default=1, help="(VNF) Set the number of virtual CPUs in a vdu. Default 1"
)
@click.option(
    "--memory",
    default=1024,
    help="(VNF) Set the memory size (MB) of the vdu. Default 1024",
)
@click.option(
    "--storage", default=10, help="(VNF) Set the disk size (GB) of the vdu. Default 10"
)
@click.option(
    "--interfaces",
    default=0,
    help="(VNF) Set the number of additional interfaces apart from the management interface. Default 0",
)
@click.option(
    "--vendor", default="OSM", help='(NS/VNF) Set the descriptor vendor. Default "OSM"'
)
@click.option(
    "--override",
    default=False,
    is_flag=True,
    help="(NS/VNF/NST) Flag for overriding the package if exists.",
)
@click.option(
    "--detailed",
    is_flag=True,
    default=False,
    help="(NS/VNF/NST) Flag for generating descriptor .yaml with all possible commented options",
)
@click.option(
    "--netslice-subnets", default=1, help="(NST) Number of netslice subnets. Default 1"
)
@click.option(
    "--netslice-vlds", default=1, help="(NST) Number of netslice vlds. Default 1"
)
@click.option(
    "--old",
    default=False,
    is_flag=True,
    help="Flag to create a descriptor using the previous OSM format (pre SOL006, OSM<9)",
)
@click.pass_context
def package_create(
    ctx,
    package_type,
    base_directory,
    package_name,
    override,
    image,
    vdus,
    vcpu,
    memory,
    storage,
    interfaces,
    vendor,
    detailed,
    netslice_subnets,
    netslice_vlds,
    old,
):
    """
    Creates an OSM NS, VNF, NST package

    \b
    PACKAGE_TYPE: Package to be created: NS, VNF or NST.
    PACKAGE_NAME: Name of the package to create the folder with the content.
    """

    logger.debug("")
    utils.check_client_version(ctx.obj, ctx.command.name)
    print(
        "Creating the {} structure: {}/{}".format(
            package_type.upper(), base_directory, package_name
        )
    )
    resp = ctx.obj.package_tool.create(
        package_type,
        base_directory,
        package_name,
        override=override,
        image=image,
        vdus=vdus,
        vcpu=vcpu,
        memory=memory,
        storage=storage,
        interfaces=interfaces,
        vendor=vendor,
        detailed=detailed,
        netslice_subnets=netslice_subnets,
        netslice_vlds=netslice_vlds,
        old=old,
    )
    print(resp)


@click.command(
    name="package-validate", short_help="Validate descriptors given a base directory"
)
@click.argument("base-directory", default=".", required=False)
@click.option(
    "--recursive/--no-recursive",
    default=True,
    help="The activated recursive option will validate the yaml files"
    " within the indicated directory and in its subdirectories",
)
@click.option(
    "--old",
    is_flag=True,
    default=False,
    help="Validates also the descriptors using the previous OSM format (pre SOL006)",
)
@click.pass_context
def package_validate(ctx, base_directory, recursive, old):
    """
    Validate descriptors given a base directory.

    \b
    BASE_DIRECTORY: Base folder for NS, VNF or NST package.
    """
    logger.debug("")
    utils.check_client_version(ctx.obj, ctx.command.name)
    results = ctx.obj.package_tool.validate(base_directory, recursive, old)
    table = PrettyTable()
    table.field_names = ["TYPE", "PATH", "VALID", "ERROR"]
    # Print the dictionary generated by the validation function
    for result in results:
        table.add_row(
            [result["type"], result["path"], result["valid"], result["error"]]
        )
    table.sortby = "VALID"
    table.align["PATH"] = "l"
    table.align["TYPE"] = "l"
    table.align["ERROR"] = "l"
    print(table)


@click.command(
    name="package-translate", short_help="Translate descriptors given a base directory"
)
@click.argument("base-directory", default=".", required=False)
@click.option(
    "--recursive/--no-recursive",
    default=True,
    help="The activated recursive option will translate the yaml files"
    " within the indicated directory and in its subdirectories",
)
@click.option(
    "--dryrun",
    is_flag=True,
    default=False,
    help="Do not translate yet, only make a dry-run to test translation",
)
@click.pass_context
def package_translate(ctx, base_directory, recursive, dryrun):
    """
    Translate descriptors given a base directory.

    \b
    BASE_DIRECTORY: Stub folder for NS, VNF or NST package.
    """
    logger.debug("")
    utils.check_client_version(ctx.obj, ctx.command.name)
    results = ctx.obj.package_tool.translate(base_directory, recursive, dryrun)
    table = PrettyTable()
    table.field_names = [
        "CURRENT TYPE",
        "NEW TYPE",
        "PATH",
        "VALID",
        "TRANSLATED",
        "ERROR",
    ]
    # Print the dictionary generated by the validation function
    for result in results:
        table.add_row(
            [
                result["current type"],
                result["new type"],
                result["path"],
                result["valid"],
                result["translated"],
                result["error"],
            ]
        )
    table.sortby = "TRANSLATED"
    table.align["PATH"] = "l"
    table.align["TYPE"] = "l"
    table.align["ERROR"] = "l"
    print(table)


@click.command(name="package-build", short_help="Build the tar.gz of the package")
@click.argument("package-folder")
@click.option(
    "--skip-validation", default=False, is_flag=True, help="skip package validation"
)
@click.option(
    "--skip-charm-build",
    default=False,
    is_flag=True,
    help="the charm will not be compiled, it is assumed to already exist",
)
@click.pass_context
def package_build(ctx, package_folder, skip_validation, skip_charm_build):
    """
    Build the package NS, VNF given the package_folder.

    \b
    PACKAGE_FOLDER: Folder of the NS, VNF or NST to be packaged
    """
    logger.debug("")
    utils.check_client_version(ctx.obj, ctx.command.name)
    results = ctx.obj.package_tool.build(
        package_folder,
        skip_validation=skip_validation,
        skip_charm_build=skip_charm_build,
    )
    print(results)


@click.command(
    name="descriptor-translate",
    short_help="Translate input descriptor file from Rel EIGHT OSM descriptors to SOL006 and prints in standard output",
)
@click.argument("descriptor-file", required=True)
@click.pass_context
def descriptor_translate(ctx, descriptor_file):
    """
    Translate input descriptor.

    \b
    DESCRIPTOR_FILE: Descriptor file for NS, VNF or Network Slice.
    """
    logger.debug("")
    utils.check_client_version(ctx.obj, ctx.command.name)
    result = ctx.obj.package_tool.descriptor_translate(descriptor_file)
    print(result)


# TODO: check if this command should be here. It is more related to nspkg and nfpkg
@click.command(name="upload-package", short_help="uploads a VNF package or NS package")
@click.argument("filename")
@click.option(
    "--skip-charm-build",
    default=False,
    is_flag=True,
    help="the charm will not be compiled, it is assumed to already exist",
)
@click.pass_context
def upload_package(ctx, filename, skip_charm_build):
    """uploads a vnf package or ns package

    filename: vnf or ns package folder, or vnf or ns package file (tar.gz)
    """
    logger.debug("")
    ctx.obj.package.upload(filename, skip_charm_build=skip_charm_build)
