########################################################################
# Copyright ETSI Contributors and Others.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import asyncio
from osm_common.temporal_task_queues.task_queues_mappings import LCM_TASK_QUEUE
from osm_common.temporal.workflows.vim import (
    VimCreateWorkflow,
    VimDeleteWorkflow,
    VimUpdateWorkflow,
)
from osm_common.temporal.workflows.lcm import NsNoOpWorkflow
from osm_common.temporal.workflows.ns import NsInstantiateWorkflow
from osm_common.wftemporal import WFTemporal


class NbiTemporal:
    workflow_mappings = {
        "created": VimCreateWorkflow,
        "edited": VimUpdateWorkflow,
        "delete": VimDeleteWorkflow,
    }

    def start_vim_workflow(self, action: str, content: dict) -> None:
        vim_uuid = content["_id"]
        # Derive the operation id (although for a create it will always be 0)
        op_id = content["op_id"].split(":")[1]
        workflow_object = NbiTemporal.workflow_mappings[action]
        workflow_data = workflow_object.Input(vim_uuid, op_id)
        asyncio.run(
            WFTemporal(logger_name="nbi.vim_workflow").start_workflow(
                task_queue=LCM_TASK_QUEUE,
                workflow_name=workflow_object.__name__,
                workflow_data=workflow_data,
                id=vim_uuid,
            )
        )

    lcm_workflow_mappings = {
        "instantiate": NsInstantiateWorkflow,
        "terminate": NsNoOpWorkflow,
        "vca_status_refresh": NsNoOpWorkflow,
        "action": NsNoOpWorkflow,
        "update": NsNoOpWorkflow,
        "scale": NsNoOpWorkflow,
        "heal": NsNoOpWorkflow,
        "migrate": NsNoOpWorkflow,
        "verticalscale": NsNoOpWorkflow,
        "deleted": NsNoOpWorkflow,
        "vnf_terminated": NsNoOpWorkflow,
        "policy_updated": NsNoOpWorkflow,
        "terminated": NsNoOpWorkflow,
        "instantiated": NsNoOpWorkflow,
        "scaled": NsNoOpWorkflow,
        "healed": NsNoOpWorkflow,
        "actioned": NsNoOpWorkflow,
        "updated": NsNoOpWorkflow,
        "migrated": NsNoOpWorkflow,
        "verticalscaled": NsNoOpWorkflow,
    }

    def start_ns_workflow(self, nslcmop: dict) -> None:
        workflow_object = NbiTemporal.lcm_workflow_mappings[nslcmop["lcmOperationType"]]
        asyncio.run(
            WFTemporal(logger_name="nbi.lcm_workflow").start_workflow(
                task_queue=LCM_TASK_QUEUE,
                workflow_name=workflow_object.__name__,
                workflow_data=workflow_object.Input(nslcmop=nslcmop),
                id=nslcmop["nsInstanceId"],
            )
        )
