##
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# For those usages not covered by the Apache License, Version 2.0 please
# contact: alfonso.tiernosepulveda@telefonica.com
##

import asynctest
import logging

from asynctest.mock import Mock
from osm_common.dbmemory import DbMemory
from osm_common.fslocal import FsLocal
from n2vc.k8s_helm_conn import K8sHelmConnector

__author__ = "Isabel Lloret <illoret@indra.es>"


class TestK8sHelmConn(asynctest.TestCase):
    logging.basicConfig(level=logging.DEBUG)
    logger = logging.getLogger(__name__)
    logger.setLevel(logging.DEBUG)

    async def setUp(self):
        self.db = Mock(DbMemory())
        self.fs = asynctest.Mock(FsLocal())
        self.fs.path = "./tmp/"
        self.namespace = "testk8s"
        self.service_account = "osm"
        self.cluster_id = "helm_cluster_id"
        self.cluster_uuid = self.cluster_id
        # pass fake kubectl and helm commands to make sure it does not call actual commands
        K8sHelmConnector._check_file_exists = asynctest.Mock(return_value=True)
        K8sHelmConnector._local_async_exec = asynctest.CoroutineMock(
            return_value=(0, "")
        )
        cluster_dir = self.fs.path + self.cluster_id
        self.kube_config = self.fs.path + self.cluster_id + "/.kube/config"
        self.helm_home = self.fs.path + self.cluster_id + "/.helm"
        self.env = {
            "HELM_HOME": "{}/.helm".format(cluster_dir),
            "KUBECONFIG": "{}/.kube/config".format(cluster_dir),
        }
        self.helm_conn = K8sHelmConnector(self.fs, self.db, log=self.logger)
        self.logger.debug("Set up executed")

    @asynctest.fail_on(active_handles=True)
    async def test_init_env(self):
        # TODO
        pass

    @asynctest.fail_on(active_handles=True)
    async def test_repo_add(self):
        repo_name = "bitnami"
        repo_url = "https://charts.bitnami.com/bitnami"
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        await self.helm_conn.repo_add(self.cluster_uuid, repo_name, repo_url)

        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        self.assertEqual(
            self.helm_conn._local_async_exec.call_count,
            2,
            "local_async_exec expected 2 calls, called {}".format(
                self.helm_conn._local_async_exec.call_count
            ),
        )

        repo_update_command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm repo update {}"
        ).format(repo_name)
        repo_add_command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm repo add {} {}"
        ).format(repo_name, repo_url)
        calls = self.helm_conn._local_async_exec.call_args_list
        call0_kargs = calls[0][1]
        self.assertEqual(
            call0_kargs.get("command"),
            repo_add_command,
            "Invalid repo add command: {}".format(call0_kargs.get("command")),
        )
        self.assertEqual(
            call0_kargs.get("env"),
            self.env,
            "Invalid env for add command: {}".format(call0_kargs.get("env")),
        )
        call1_kargs = calls[1][1]
        self.assertEqual(
            call1_kargs.get("command"),
            repo_update_command,
            "Invalid repo update command: {}".format(call1_kargs.get("command")),
        )
        self.assertEqual(
            call1_kargs.get("env"),
            self.env,
            "Invalid env for update command: {}".format(call1_kargs.get("env")),
        )

    @asynctest.fail_on(active_handles=True)
    async def test_repo_list(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        await self.helm_conn.repo_list(self.cluster_uuid)

        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        command = "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm repo list --output yaml"
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

    @asynctest.fail_on(active_handles=True)
    async def test_repo_remove(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        repo_name = "bitnami"
        await self.helm_conn.repo_remove(self.cluster_uuid, repo_name)

        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        command = "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm repo remove {}".format(
            repo_name
        )
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=True
        )

    @asynctest.fail_on(active_handles=True)
    async def test_install(self):
        kdu_model = "stable/openldap:1.2.2"
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._status_kdu = asynctest.CoroutineMock(return_value=None)
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.generate_kdu_instance_name = Mock(return_value=kdu_instance)

        await self.helm_conn.install(
            self.cluster_uuid,
            kdu_model,
            kdu_instance,
            atomic=True,
            namespace=self.namespace,
            db_dict=db_dict,
        )

        self.helm_conn.fs.sync.assert_has_calls(
            [
                asynctest.call(from_path=self.cluster_id),
                asynctest.call(from_path=self.cluster_id),
            ]
        )
        self.helm_conn.fs.reverse_sync.assert_has_calls(
            [
                asynctest.call(from_path=self.cluster_id),
                asynctest.call(from_path=self.cluster_id),
            ]
        )
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="install",
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm install "
            "--atomic --output yaml   --timeout 300 "
            "--name=stable-openldap-0005399828 --namespace testk8s stable/openldap "
            "--version 1.2.2"
        )
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

    @asynctest.fail_on(active_handles=True)
    async def test_upgrade_force_true(self):
        kdu_model = "stable/openldap:1.2.3"
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        instance_info = {
            "chart": "openldap-1.2.2",
            "name": kdu_instance,
            "namespace": self.namespace,
            "revision": 1,
            "status": "DEPLOYED",
        }
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.get_instance_info = asynctest.CoroutineMock(
            return_value=instance_info
        )
        # TEST-1 (--force true)
        await self.helm_conn.upgrade(
            self.cluster_uuid,
            kdu_instance,
            kdu_model,
            atomic=True,
            db_dict=db_dict,
            force=True,
        )
        self.helm_conn.fs.sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_has_calls(
            [
                asynctest.call(from_path=self.cluster_id),
                asynctest.call(from_path=self.cluster_id),
            ]
        )
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="upgrade",
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm upgrade --namespace testk8s "
            "--atomic --output yaml  --timeout 300 --force --reuse-values stable-openldap-0005399828 stable/openldap "
            "--version 1.2.3"
        )
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )
        # TEST-2 (--force false)
        await self.helm_conn.upgrade(
            self.cluster_uuid,
            kdu_instance,
            kdu_model,
            atomic=True,
            db_dict=db_dict,
        )
        self.helm_conn.fs.sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_has_calls(
            [
                asynctest.call(from_path=self.cluster_id),
                asynctest.call(from_path=self.cluster_id),
            ]
        )
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="upgrade",
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm upgrade --namespace testk8s "
            "--atomic --output yaml  --timeout 300 --reuse-values stable-openldap-0005399828 stable/openldap "
            "--version 1.2.3"
        )
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

    @asynctest.fail_on(active_handles=True)
    async def test_upgrade_namespace(self):
        kdu_model = "stable/openldap:1.2.3"
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        instance_info = {
            "chart": "openldap-1.2.2",
            "name": kdu_instance,
            "namespace": self.namespace,
            "revision": 1,
            "status": "DEPLOYED",
        }
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.get_instance_info = asynctest.CoroutineMock(
            return_value=instance_info
        )

        await self.helm_conn.upgrade(
            self.cluster_uuid,
            kdu_instance,
            kdu_model,
            atomic=True,
            db_dict=db_dict,
            namespace="default",
        )
        self.helm_conn.fs.sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_has_calls(
            [
                asynctest.call(from_path=self.cluster_id),
                asynctest.call(from_path=self.cluster_id),
            ]
        )
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace="default",
            db_dict=db_dict,
            operation="upgrade",
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm upgrade --namespace default "
            "--atomic --output yaml  --timeout 300 --reuse-values stable-openldap-0005399828 stable/openldap "
            "--version 1.2.3"
        )
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

    @asynctest.fail_on(active_handles=True)
    async def test_scale(self):
        kdu_model = "stable/openldap:1.2.3"
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        instance_info = {
            "chart": "openldap-1.2.3",
            "name": kdu_instance,
            "namespace": self.namespace,
            "revision": 1,
            "status": "DEPLOYED",
        }
        repo_list = [
            {
                "name": "stable",
                "url": "https://kubernetes-charts.storage.googleapis.com/",
            }
        ]
        kdu_values = """
            # Default values for openldap.
            # This is a YAML-formatted file.
            # Declare variables to be passed into your templates.

            replicaCount: 1
            dummy-app:
              replicas: 2
        """

        self.helm_conn.repo_list = asynctest.CoroutineMock(return_value=repo_list)
        self.helm_conn.values_kdu = asynctest.CoroutineMock(return_value=kdu_values)
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.get_instance_info = asynctest.CoroutineMock(
            return_value=instance_info
        )

        # TEST-1
        await self.helm_conn.scale(
            kdu_instance,
            2,
            "",
            kdu_model=kdu_model,
            cluster_uuid=self.cluster_uuid,
            atomic=True,
            db_dict=db_dict,
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config "
            "/usr/bin/helm upgrade --namespace testk8s --atomic --output yaml --set replicaCount=2 "
            "--timeout 1800 --reuse-values stable-openldap-0005399828 stable/openldap "
            "--version 1.2.3"
        )
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

        # TEST-2
        await self.helm_conn.scale(
            kdu_instance,
            3,
            "dummy-app",
            kdu_model=kdu_model,
            cluster_uuid=self.cluster_uuid,
            atomic=True,
            db_dict=db_dict,
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config "
            "/usr/bin/helm upgrade --namespace testk8s --atomic --output yaml --set dummy-app.replicas=3 "
            "--timeout 1800 --reuse-values stable-openldap-0005399828 stable/openldap "
            "--version 1.2.3"
        )
        self.helm_conn._local_async_exec.assert_called_with(
            command=command, env=self.env, raise_exception_on_error=False
        )
        self.helm_conn.fs.reverse_sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="scale",
        )

    @asynctest.fail_on(active_handles=True)
    async def test_rollback(self):
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        instance_info = {
            "chart": "openldap-1.2.3",
            "name": kdu_instance,
            "namespace": self.namespace,
            "revision": 2,
            "status": "DEPLOYED",
        }
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.get_instance_info = asynctest.CoroutineMock(
            return_value=instance_info
        )

        await self.helm_conn.rollback(
            self.cluster_uuid, kdu_instance=kdu_instance, revision=1, db_dict=db_dict
        )
        self.helm_conn.fs.sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        self.helm_conn._store_status.assert_called_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="rollback",
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config "
            "/usr/bin/helm rollback stable-openldap-0005399828 1 --wait"
        )
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=False
        )

    @asynctest.fail_on(active_handles=True)
    async def test_uninstall(self):
        kdu_instance = "stable-openldap-0005399828"
        instance_info = {
            "chart": "openldap-1.2.2",
            "name": kdu_instance,
            "namespace": self.namespace,
            "revision": 3,
            "status": "DEPLOYED",
        }
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        self.helm_conn._store_status = asynctest.CoroutineMock()
        self.helm_conn.get_instance_info = asynctest.CoroutineMock(
            return_value=instance_info
        )

        await self.helm_conn.uninstall(self.cluster_uuid, kdu_instance)
        self.helm_conn.fs.sync.assert_called_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        command = "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm delete --purge  {}".format(
            kdu_instance
        )
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=True
        )

    @asynctest.fail_on(active_handles=True)
    async def test_get_services(self):
        kdu_instance = "test_services_1"
        service = {"name": "testservice", "type": "LoadBalancer"}
        self.helm_conn._local_async_exec_pipe = asynctest.CoroutineMock(
            return_value=("", 0)
        )
        self.helm_conn._parse_services = Mock(return_value=["testservice"])
        self.helm_conn._get_service = asynctest.CoroutineMock(return_value=service)

        services = await self.helm_conn.get_services(
            self.cluster_uuid, kdu_instance, self.namespace
        )
        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        self.helm_conn._parse_services.assert_called_once()
        command1 = "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm get manifest {} ".format(
            kdu_instance
        )
        command2 = "/usr/bin/kubectl get --namespace={} -f -".format(self.namespace)
        self.helm_conn._local_async_exec_pipe.assert_called_once_with(
            command1, command2, env=self.env, raise_exception_on_error=True
        )
        self.assertEqual(
            services, [service], "Invalid service returned from get_service"
        )

    @asynctest.fail_on(active_handles=True)
    async def test_get_service(self):
        service_name = "service1"

        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))
        await self.helm_conn.get_service(
            self.cluster_uuid, service_name, self.namespace
        )

        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        command = (
            "/usr/bin/kubectl --kubeconfig=./tmp/helm_cluster_id/.kube/config "
            "--namespace=testk8s get service service1 -o=yaml"
        )
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=True
        )

    @asynctest.fail_on(active_handles=True)
    async def test_inspect_kdu(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        kdu_model = "stable/openldap:1.2.4"
        repo_url = "https://kubernetes-charts.storage.googleapis.com/"
        await self.helm_conn.inspect_kdu(kdu_model, repo_url)

        command = (
            "/usr/bin/helm inspect  openldap --repo "
            "https://kubernetes-charts.storage.googleapis.com/ "
            "--version 1.2.4"
        )
        self.helm_conn._local_async_exec.assert_called_with(command=command)

    @asynctest.fail_on(active_handles=True)
    async def test_help_kdu(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        kdu_model = "stable/openldap:1.2.4"
        repo_url = "https://kubernetes-charts.storage.googleapis.com/"
        await self.helm_conn.help_kdu(kdu_model, repo_url)

        command = (
            "/usr/bin/helm inspect readme openldap --repo "
            "https://kubernetes-charts.storage.googleapis.com/ "
            "--version 1.2.4"
        )
        self.helm_conn._local_async_exec.assert_called_with(command=command)

    @asynctest.fail_on(active_handles=True)
    async def test_values_kdu(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        kdu_model = "stable/openldap:1.2.4"
        repo_url = "https://kubernetes-charts.storage.googleapis.com/"
        await self.helm_conn.values_kdu(kdu_model, repo_url)

        command = (
            "/usr/bin/helm inspect values openldap --repo "
            "https://kubernetes-charts.storage.googleapis.com/ "
            "--version 1.2.4"
        )
        self.helm_conn._local_async_exec.assert_called_with(command=command)

    @asynctest.fail_on(active_handles=True)
    async def test_get_values_kdu(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        kdu_instance = "stable-openldap-0005399828"
        await self.helm_conn.get_values_kdu(
            kdu_instance, self.namespace, self.env["KUBECONFIG"]
        )

        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm get values "
            "stable-openldap-0005399828 --output yaml"
        )
        self.helm_conn._local_async_exec.assert_called_with(command=command)

    @asynctest.fail_on(active_handles=True)
    async def test_instances_list(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        await self.helm_conn.instances_list(self.cluster_uuid)
        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.reverse_sync.assert_called_once_with(
            from_path=self.cluster_id
        )
        command = "/usr/bin/helm list --output yaml"
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command, env=self.env, raise_exception_on_error=True
        )

    @asynctest.fail_on(active_handles=True)
    async def test_status_kdu(self):
        kdu_instance = "stable-openldap-0005399828"
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        await self.helm_conn._status_kdu(
            self.cluster_id, kdu_instance, self.namespace, yaml_format=True
        )
        command = (
            "env KUBECONFIG=./tmp/helm_cluster_id/.kube/config /usr/bin/helm status {} --output yaml"
        ).format(kdu_instance)
        self.helm_conn._local_async_exec.assert_called_once_with(
            command=command,
            env=self.env,
            raise_exception_on_error=True,
            show_error_log=False,
        )

    @asynctest.fail_on(active_handles=True)
    async def test_store_status(self):
        kdu_instance = "stable-openldap-0005399828"
        db_dict = {}
        status = {
            "info": {
                "description": "Install complete",
                "status": {
                    "code": "1",
                    "notes": "The openldap helm chart has been installed",
                },
            }
        }
        self.helm_conn._status_kdu = asynctest.CoroutineMock(return_value=status)
        self.helm_conn.write_app_status_to_db = asynctest.CoroutineMock(
            return_value=status
        )

        await self.helm_conn._store_status(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            db_dict=db_dict,
            operation="install",
        )
        self.helm_conn._status_kdu.assert_called_once_with(
            cluster_id=self.cluster_id,
            kdu_instance=kdu_instance,
            namespace=self.namespace,
            yaml_format=False,
        )
        self.helm_conn.write_app_status_to_db.assert_called_once_with(
            db_dict=db_dict,
            status="Install complete",
            detailed_status=str(status),
            operation="install",
        )

    @asynctest.fail_on(active_handles=True)
    async def test_reset_uninstall_false(self):
        self.helm_conn._uninstall_sw = asynctest.CoroutineMock()

        await self.helm_conn.reset(self.cluster_uuid, force=False, uninstall_sw=False)
        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.file_delete.assert_called_once_with(
            self.cluster_id, ignore_non_exist=True
        )
        self.helm_conn._uninstall_sw.assert_not_called()

    @asynctest.fail_on(active_handles=True)
    async def test_reset_uninstall(self):
        kdu_instance = "stable-openldap-0021099429"
        instances = [
            {
                "app_version": "2.4.48",
                "chart": "openldap-1.2.3",
                "name": kdu_instance,
                "namespace": self.namespace,
                "revision": "1",
                "status": "deployed",
                "updated": "2020-10-30 11:11:20.376744191 +0000 UTC",
            }
        ]
        self.helm_conn._get_namespace = Mock(return_value=self.namespace)
        self.helm_conn._uninstall_sw = asynctest.CoroutineMock()
        self.helm_conn.instances_list = asynctest.CoroutineMock(return_value=instances)
        self.helm_conn.uninstall = asynctest.CoroutineMock()

        await self.helm_conn.reset(self.cluster_uuid, force=True, uninstall_sw=True)
        self.helm_conn.fs.sync.assert_called_once_with(from_path=self.cluster_id)
        self.helm_conn.fs.file_delete.assert_called_once_with(
            self.cluster_id, ignore_non_exist=True
        )
        self.helm_conn._get_namespace.assert_called_once_with(
            cluster_uuid=self.cluster_uuid
        )
        self.helm_conn.instances_list.assert_called_once_with(
            cluster_uuid=self.cluster_uuid
        )
        self.helm_conn.uninstall.assert_called_once_with(
            cluster_uuid=self.cluster_uuid, kdu_instance=kdu_instance
        )
        self.helm_conn._uninstall_sw.assert_called_once_with(
            cluster_id=self.cluster_id, namespace=self.namespace
        )

    @asynctest.fail_on(active_handles=True)
    async def test_uninstall_sw_namespace(self):
        self.helm_conn._local_async_exec = asynctest.CoroutineMock(return_value=("", 0))

        await self.helm_conn._uninstall_sw(self.cluster_id, self.namespace)
        calls = self.helm_conn._local_async_exec.call_args_list
        self.assertEqual(
            len(calls), 3, "To uninstall should have executed three commands"
        )
        call0_kargs = calls[0][1]
        command_0 = "/usr/bin/helm --kubeconfig={} --home={} reset".format(
            self.kube_config, self.helm_home
        )
        self.assertEqual(
            call0_kargs,
            {"command": command_0, "raise_exception_on_error": True, "env": self.env},
            "Invalid args for first call to local_exec",
        )
        call1_kargs = calls[1][1]
        command_1 = (
            "/usr/bin/kubectl --kubeconfig={} delete "
            "clusterrolebinding.rbac.authorization.k8s.io/osm-tiller-cluster-rule".format(
                self.kube_config
            )
        )
        self.assertEqual(
            call1_kargs,
            {"command": command_1, "raise_exception_on_error": False, "env": self.env},
            "Invalid args for second call to local_exec",
        )
        call2_kargs = calls[2][1]
        command_2 = (
            "/usr/bin/kubectl --kubeconfig={} --namespace {} delete "
            "serviceaccount/{}".format(
                self.kube_config, self.namespace, self.service_account
            )
        )
        self.assertEqual(
            call2_kargs,
            {"command": command_2, "raise_exception_on_error": False, "env": self.env},
            "Invalid args for third call to local_exec",
        )
