#######################################################################################
# Copyright ETSI Contributors and Others.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import asyncio
import asynctest
import uuid

from asynctest import CoroutineMock, Mock, patch
from copy import deepcopy
from datetime import timedelta
from parameterized import parameterized
from temporalio import activity, workflow
from temporalio.client import WorkflowFailureError
from temporalio.exceptions import (
    ActivityError,
    ApplicationError,
    ChildWorkflowError,
    RetryState,
)
from temporalio.testing import WorkflowEnvironment
from temporalio.worker import Worker
from typing import Any

from osm_common.temporal.activities.vnf import (
    ChangeVnfInstantiationState,
    ChangeVnfState,
    DeleteVnfRecord,
    GetTaskQueue,
    GetVimCloud,
    GetVnfDescriptor,
    GetVnfRecord,
    SendNotificationForVnf,
    SetVnfModel,
)
from osm_common.temporal.dataclasses_common import (
    CharmInfo,
    VduComputeConstraints,
)
from osm_common.temporal.states import (
    VnfInstantiationState,
    VnfState,
)
from osm_common.temporal_task_queues.task_queues_mappings import LCM_TASK_QUEUE
from osm_common.temporal.workflows.vdu import (
    VduInstantiateWorkflow,
    VduTerminateWorkflow,
)
from osm_common.temporal.workflows.vnf import (
    VnfPrepareWorkflow,
    VnfTerminateWorkflow,
)
from osm_lcm.temporal.vnf_workflows import (
    VnfDeleteWorkflowImpl,
    VnfInstantiateWorkflowImpl,
    VnfPrepareWorkflowImpl,
    VnfTerminateWorkflowImpl,
)

SANDBOXED = False
DEBUG_MODE = True
TASK_TIMEOUT = timedelta(seconds=5)
EXECUTION_TIMEOUT = timedelta(seconds=10)

# The variables used in the tests
model_name = "my-model-name"
cloud = "microk8s"
juju_task_queue = "juju_task_queue"
vnfr_uuid = "9f472177-95c0-4335-b357-5cdc17a79965"
vim_account_id = "9b0bedc3-ea8e-42fd-acc9-dd03f4dee73c"
vdu_id = "hackfest_basic-VM"
vnf_index = "vnf-profile-id"
sample_vnfr = {
    "_id": vnfr_uuid,
    "id": vnfr_uuid,
    "nsr-id-ref": "dcf4c922-5a73-41bf-a6ca-870c22d6336c",
    "vnfd-ref": "jar_vnfd_scalable",
    "vnfd-id": "f1b38eac-190c-485d-9a74-c6e169c929d8",
    "vim-account-id": vim_account_id,
    "namespace": model_name,
    "member-vnf-index-ref": vnf_index,
}
vnf_config = {
    "member-vnf-index": vnf_index,
    "vdu": [
        {
            "id": vdu_id,
            "configurable-properties": {
                "config::redirect-map": "https://osm.instantiation.params"
            },
        }
    ],
}
app_config = {"domain_name1": "osm.org", "domain_name2": "osm.com"}
config = [
    {"key": "config::domain_name1", "value": "osm.org"},
    {"key": "config::domain_name2", "value": "osm.com"},
    {"key": "track", "value": "latest"},
    {"key": "channel", "value": "stable"},
]
vdu = {
    "id": vdu_id,
    "name": "hackfest_basic-VM",
    "sw-image-desc": "ubuntu20.04",
    "virtual-compute-desc": "hackfest_basic-VM-compute",
    "virtual-storage-desc": ["hackfest_basic-VM-storage"],
    "configurable-properties": config,
}
vnfd_id = "97784f19-d254-4252-946c-cf92d85443ca"
flavor_1 = {
    "id": "compute-id-1",
    "virtual-memory": {"size": "4"},
    "virtual-cpu": {"cpu-architecture": "x86", "num-virtual-cpu": 2},
}
flavor_2 = {
    "id": "compute-id-2",
    "virtual-cpu": {"cpu-architecture": "x86", "num-virtual-cpu": 2},
}
sample_vnfd = {
    "_id": vnfd_id,
    "id": "sol006-vnf",
    "provider": "Canonical",
    "product-name": "test-vnf",
    "software-version": "1.0",
    "vdu": [vdu],
    "virtual-compute-desc": [flavor_1, flavor_2],
    "sw-image-desc": [
        {
            "id": "ubuntu20.04",
            "image": "ubuntu20.04",
            "name": "ubuntu20.04",
            "version": "1.0",
        }
    ],
}
sample_charm_info = CharmInfo(app_name="my-app", channel="latest", entity_url="my-url")
sample_constraints = VduComputeConstraints(cores=1, mem=1)
sample_vdu_instantiate_input = VduInstantiateWorkflow.Input(
    vim_uuid=vim_account_id,
    model_name=model_name,
    charm_info=sample_charm_info,
    constraints=sample_constraints,
    cloud=cloud,
    config={},
)
sample_vdu_instantiate_wf_id_1 = "vdu-instantiate-workflow-id-1"
sample_vdu_instantiate_input_with_config = VduInstantiateWorkflow.Input(
    vim_uuid=vim_account_id,
    model_name=model_name,
    charm_info=sample_charm_info,
    constraints=sample_constraints,
    cloud=cloud,
    config=app_config,
)


class TestException(Exception):
    pass


class GetApplicationConfigException(Exception):
    pass


class GetVduInstantiationParamsException(Exception):
    pass


class GetCharmInfoException(Exception):
    pass


class GetComputeConstraintsException(Exception):
    pass


class MockActivity:
    def __init__(self, *args: Any, **kwargs: Any):
        self.tracker = Mock(return_value=None)

    async def __call__(self, *args: Any, **kwargs: Any) -> Any:
        return self.tracker(*args, **kwargs)

    def __setattr__(self, __name: str, __value: Any) -> None:
        if __name in ("return_value", "side_effect"):
            self.tracker.__setattr__(__name, __value)
        else:
            super().__setattr__(__name, __value)


# Mock Activities
@activity.defn(name=SetVnfModel.__name__)
async def mock_set_vnf_model(set_vnf_model_input: SetVnfModel.Input) -> None:
    pass


@activity.defn(name=SetVnfModel.__name__)
async def mock_set_vnf_model_raise_exception(
    set_vnf_model_input: SetVnfModel.Input,
) -> None:
    raise TestException(f"{SetVnfModel.__name__} failed.")


@activity.defn(name=GetTaskQueue.__name__)
async def mock_get_task_queue(
    get_task_queue_input: GetTaskQueue.Input,
) -> GetTaskQueue.Output:
    return GetTaskQueue.Output(LCM_TASK_QUEUE)


@activity.defn(name=GetTaskQueue.__name__)
async def mock_get_different_task_queue(
    get_task_queue_input: GetTaskQueue.Input,
) -> GetTaskQueue.Output:
    return GetTaskQueue.Output(juju_task_queue)


@activity.defn(name=GetTaskQueue.__name__)
async def mock_get_task_queue_raise_exception(
    get_task_queue_input: GetTaskQueue.Input,
) -> GetTaskQueue.Output:
    raise TestException(f"{GetTaskQueue.__name__} failed.")


@activity.defn(name=GetVnfDescriptor.__name__)
class MockGetVnfDescriptor(MockActivity):
    def __init__(self, *args: Any, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self.tracker.return_value = GetVnfDescriptor.Output(vnfd=sample_vnfd)


@activity.defn(name=ChangeVnfInstantiationState.__name__)
class MockChangeVnfInstantiationState(MockActivity):
    pass


@activity.defn(name=GetVnfDescriptor.__name__)
async def mock_get_vnf_descriptor_empty_output(
    get_vnf_descriptor_input: GetVnfDescriptor.Input,
) -> GetVnfDescriptor.Output:
    return GetVnfDescriptor.Output(vnfd={})


@activity.defn(name=GetVnfDescriptor.__name__)
async def mock_get_vnf_descriptor_raise_exception(
    get_vnf_descriptor_input: GetVnfDescriptor.Input,
) -> GetVnfDescriptor.Output:
    raise TestException(f"{GetVnfDescriptor.__name__} failed.")


@activity.defn(name=GetVnfRecord.__name__)
class MockGetVnfRecord(MockActivity):
    def __init__(self, *args: Any, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self.tracker.return_value = GetVnfRecord.Output(vnfr=sample_vnfr)


@activity.defn(name=GetVnfRecord.__name__)
async def mock_get_vnf_record_not_instantiated_state(
    get_vnf_record_input: GetVnfRecord.Input,
) -> GetVnfRecord.Output:
    vnfr = deepcopy(sample_vnfr)
    vnfr["instantiationState"] = VnfInstantiationState.NOT_INSTANTIATED.name
    return GetVnfRecord.Output(vnfr=vnfr)


@activity.defn(name=GetVnfRecord.__name__)
async def mock_get_vnf_record_instantiated_state(
    get_vnf_record_input: GetVnfRecord.Input,
) -> GetVnfRecord.Output:
    vnfr = deepcopy(sample_vnfr)
    vnfr["instantiationState"] = VnfInstantiationState.INSTANTIATED.name
    return GetVnfRecord.Output(vnfr=vnfr)


@activity.defn(name=GetVnfRecord.__name__)
async def mock_get_vnf_record_raise_exception(
    get_vnf_record_input: GetVnfRecord.Input,
) -> GetVnfRecord.Output:
    raise TestException(f"{GetVnfRecord.__name__} failed.")


@activity.defn(name=GetVimCloud.__name__)
async def mock_get_vim_cloud(
    get_vim_cloud_input: GetVimCloud.Input,
) -> GetVimCloud.Output:
    return GetVimCloud.Output(cloud=cloud)


@activity.defn(name=GetVimCloud.__name__)
async def mock_get_vim_cloud_raise_exception(
    get_vim_cloud_input: GetVimCloud.Input,
) -> GetVimCloud.Output:
    raise TestException(f"{GetVimCloud.__name__} failed.")


# Mock Workflows
@workflow.defn(name=VnfPrepareWorkflow.__name__, sandboxed=SANDBOXED)
class MockPrepareVnfWorkflow:
    @workflow.run
    async def run(self, input: VnfPrepareWorkflow.Input) -> None:
        pass


@workflow.defn(name=VnfPrepareWorkflow.__name__, sandboxed=SANDBOXED)
class MockPrepareVnfWorkflowFailed:
    @workflow.run
    async def run(self, workflow_input: VnfPrepareWorkflow.Input) -> None:
        raise ChildWorkflowError(
            message=f"{VnfPrepareWorkflow.__name__} child workflow failed.",
            namespace="default",
            workflow_id="123",
            run_id="1",
            workflow_type=VnfPrepareWorkflow.__name__,
            initiated_event_id=0,
            started_event_id=0,
            retry_state=RetryState.NON_RETRYABLE_FAILURE,
        )


@workflow.defn(name=VduInstantiateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVduInstantiateWorkflow:
    @workflow.run
    async def run(self, workflow_input: VduInstantiateWorkflow.Input) -> None:
        pass


@workflow.defn(name=VduInstantiateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVduInstantiateWorkflowSlow:
    @workflow.run
    async def run(self, workflow_input: VduInstantiateWorkflow.Input) -> None:
        await asyncio.sleep(0.5)


@workflow.defn(name=VduTerminateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVduTerminateWorkflow:
    @workflow.run
    async def run(self, workflow_input: VduTerminateWorkflow.Input) -> None:
        pass


@workflow.defn(name=VduTerminateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVduTerminateWorkflowRaisesException:
    @workflow.run
    async def run(self, workflow_input: VduTerminateWorkflow.Input) -> None:
        raise TestException(f"{self.__class__.__name__} failed.")


@workflow.defn(name=VduInstantiateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVduInstantiateWorkflowFailed:
    @workflow.run
    async def run(self, workflow_input: VduInstantiateWorkflow.Input) -> None:
        raise ChildWorkflowError(
            message=f"{VduInstantiateWorkflow.__name__} child workflow failed.",
            namespace="default",
            workflow_id="123",
            run_id="1",
            workflow_type=VduInstantiateWorkflow.__name__,
            initiated_event_id=0,
            started_event_id=0,
            retry_state=RetryState.NON_RETRYABLE_FAILURE,
        )


class TestVnfPrepareWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client
        self.task_queue = LCM_TASK_QUEUE
        self.wf_input = VnfPrepareWorkflowImpl.Input(
            vnfr_uuid=vnfr_uuid, model_name=model_name
        )

    async def test_vnf_prepare_workflow__successful(self):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfPrepareWorkflowImpl],
                activities=[mock_set_vnf_model],
            ):
                await self.client.execute_workflow(
                    VnfPrepareWorkflowImpl.run,
                    arg=self.wf_input,
                    id=f"{VnfPrepareWorkflow.__name__}-{self.wf_input.vnfr_uuid}",
                    task_queue=self.task_queue,
                    task_timeout=TASK_TIMEOUT,
                    execution_timeout=EXECUTION_TIMEOUT,
                )

    async def test_vnf_prepare_workflow__activity_set_vnf_model_fails__raise_activity_error(
        self,
    ):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfPrepareWorkflowImpl],
                activities=[mock_set_vnf_model_raise_exception],
            ):
                with self.assertRaises(WorkflowFailureError) as err:
                    await self.client.execute_workflow(
                        VnfPrepareWorkflowImpl.run,
                        arg=self.wf_input,
                        id=f"{VnfPrepareWorkflow.__name__}-{self.wf_input.vnfr_uuid}",
                        task_queue=self.task_queue,
                        task_timeout=TASK_TIMEOUT,
                        execution_timeout=EXECUTION_TIMEOUT,
                    )
        self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_prepare_workflow__activity_set_vnf_model_fails__get_expected_error_details(
        self,
    ):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfPrepareWorkflowImpl],
                activities=[mock_set_vnf_model_raise_exception],
            ):
                with self.assertRaises(WorkflowFailureError) as err:
                    await self.client.execute_workflow(
                        VnfPrepareWorkflowImpl.run,
                        arg=self.wf_input,
                        id=f"{VnfPrepareWorkflow.__name__}-{self.wf_input.vnfr_uuid}",
                        task_queue=self.task_queue,
                        task_timeout=TASK_TIMEOUT,
                        execution_timeout=EXECUTION_TIMEOUT,
                    )
        self.assertEqual(
            str(err.exception.cause.cause),
            f"TestException: {SetVnfModel.__name__} failed.",
        )


class TestVnfInstantiateWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client
        self.mock_change_vnf_instantiation_state_tracker = Mock()
        self.mock_change_vnf_state_tracker = Mock()
        self.mock_send_notification_for_vnf_tracker = Mock()
        self.workflows = [
            VnfInstantiateWorkflowImpl,
            MockPrepareVnfWorkflow,
            MockVduInstantiateWorkflow,
        ]
        self.task_queue = LCM_TASK_QUEUE
        self.wf_input = VnfInstantiateWorkflowImpl.Input(
            vnfr_uuid=vnfr_uuid,
            model_name=model_name,
            instantiation_config={},
        )

    def get_worker(self, task_queue: str, workflows: list, activities: list) -> Worker:
        return Worker(
            self.client,
            task_queue=task_queue,
            workflows=workflows,
            activities=activities,
            debug_mode=DEBUG_MODE,
        )

    async def execute_workflow(self, wf_input):
        return await self.client.execute_workflow(
            VnfInstantiateWorkflowImpl.run,
            arg=wf_input,
            id=uuid.uuid4().hex,
            task_queue=self.task_queue,
            task_timeout=TASK_TIMEOUT,
            execution_timeout=EXECUTION_TIMEOUT,
        )

    def validate_vnf_instantiation_state_is_updated(self, state):
        call_mock_change_vnf_instantiation_state_tracker = (
            self.mock_change_vnf_instantiation_state_tracker.call_args_list
        )
        self.assertEqual(
            call_mock_change_vnf_instantiation_state_tracker[1].args[0],
            ChangeVnfInstantiationState.Input(vnfr_uuid=vnfr_uuid, state=state),
        )

    def vnf_state_is_updated(self, state):
        call_mock_change_vnf_state_tracker = (
            self.mock_change_vnf_state_tracker.call_args_list
        )
        self.assertEqual(
            call_mock_change_vnf_state_tracker[0].args[0],
            ChangeVnfState.Input(vnfr_uuid=vnfr_uuid, state=state),
        )

    @activity.defn(name=ChangeVnfState.__name__)
    async def mock_change_vnf_state(self, nf_state_input: ChangeVnfState.Input) -> None:
        self.mock_change_vnf_state_tracker(nf_state_input)

    @activity.defn(name=ChangeVnfState.__name__)
    async def mock_change_vnf_state_exception(
        self, nf_state_input: ChangeVnfState.Input
    ) -> None:
        self.mock_change_vnf_state_tracker(nf_state_input)
        raise TestException(f"{ChangeVnfState.__name__} failed.")

    @activity.defn(name=ChangeVnfInstantiationState.__name__)
    async def mock_change_vnf_instantiation_state(
        self,
        nf_instantiation_state_input: ChangeVnfInstantiationState.Input,
    ) -> None:
        self.mock_change_vnf_instantiation_state_tracker(nf_instantiation_state_input)

    @activity.defn(name=ChangeVnfInstantiationState.__name__)
    async def mock_change_vnf_instantiation_state_exception(
        self,
        nf_instantiation_state_input: ChangeVnfInstantiationState.Input,
    ) -> None:
        self.mock_change_vnf_instantiation_state_tracker(nf_instantiation_state_input)
        raise TestException(f"{ChangeVnfInstantiationState.__name__} failed.")

    @activity.defn(name=SendNotificationForVnf.__name__)
    async def mock_send_notification_for_vnf(self, input) -> None:
        self.mock_send_notification_for_vnf_tracker()

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    async def test_vnf_instantiate_workflow__successful__updates_vnf_instantiation_state(
        self, mock_get_vdu_instantiate_input
    ):
        workflows = [
            VnfInstantiateWorkflowImpl,
            MockPrepareVnfWorkflow,
            MockVduInstantiateWorkflow,
        ]
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input_with_config,
            sample_vdu_instantiate_wf_id_1,
        )
        async with self.env, self.get_worker(
            self.task_queue, workflows, activities
        ), self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        self.validate_vnf_instantiation_state_is_updated(
            VnfInstantiationState.INSTANTIATED
        )

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    async def test_vnf_instantiate_workflow__successful__updates_vnf_state(
        self, mock_get_vdu_instantiate_input
    ):
        workflows = [
            VnfInstantiateWorkflowImpl,
            MockPrepareVnfWorkflow,
            MockVduInstantiateWorkflow,
        ]
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input_with_config,
            sample_vdu_instantiate_wf_id_1,
        )
        async with self.env, self.get_worker(
            self.task_queue, workflows, activities
        ), self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        self.validate_vnf_instantiation_state_is_updated(VnfState.STARTED)

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch(
        "temporalio.workflow.execute_child_workflow",
        wraps=workflow.execute_child_workflow,
    )
    async def test_vnf_instantiate_workflow__successful__executes_child_workflow(
        self,
        mock_execute_child_workflow: CoroutineMock,
        mock_get_vdu_instantiate_input,
    ):
        workflows = [
            VnfInstantiateWorkflowImpl,
            MockPrepareVnfWorkflow,
            MockVduInstantiateWorkflow,
        ]
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input_with_config,
            sample_vdu_instantiate_wf_id_1,
        )
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        self.assertTrue(mock_execute_child_workflow.called)
        self.assertEquals(mock_execute_child_workflow.call_count, 2)
        # Check that PrepareVnfWorkflow is executed
        call_args = mock_execute_child_workflow.call_args_list[0].kwargs["workflow"]
        self.assertEquals(call_args, VnfPrepareWorkflow.__name__)
        # Check that VduInstantiateWorkflow is executed
        call_args = mock_execute_child_workflow.call_args_list[1].kwargs["workflow"]
        self.assertEquals(call_args, VduInstantiateWorkflow.__name__)

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch(
        "temporalio.workflow.execute_child_workflow",
        wraps=workflow.execute_child_workflow,
    )
    async def test_vnf_instantiate_workflow__successful__executes_child_workflows_concurrently(
        self,
        mock_execute_child_workflow: CoroutineMock,
        mock_get_vdu_instantiate_input,
    ):
        # Choose a number of child workflows that would take longer to execute than the execution timeout
        num_child_workflows = EXECUTION_TIMEOUT.seconds * 2 + 1
        workflows = [
            VnfInstantiateWorkflowImpl,
            MockPrepareVnfWorkflow,
            MockVduInstantiateWorkflowSlow,
        ]
        mock_get_vnf_descriptor = MockGetVnfDescriptor()
        mock_get_vnf_descriptor.return_value = GetVnfDescriptor.Output(
            vnfd={
                "vdu": [vdu] * num_child_workflows,
            }
        )
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            mock_get_vnf_descriptor,
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]

        # Set up the return values with a unique workflow ID for each child workflow
        mock_get_vdu_instantiate_input.side_effect = [
            (
                sample_vdu_instantiate_input_with_config,
                uuid.uuid4().hex,
            )
            for _ in range(num_child_workflows)
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)

        self.assertEquals(
            mock_execute_child_workflow.call_count, num_child_workflows + 1
        )
        # Check that PrepareVnfWorkflow is executed
        call_args = mock_execute_child_workflow.call_args_list[0].kwargs["workflow"]
        self.assertEquals(call_args, VnfPrepareWorkflow.__name__)
        # Check that VduInstantiateWorkflow is executed
        for i in range(1, num_child_workflows + 1):
            call_args = mock_execute_child_workflow.call_args_list[i].kwargs["workflow"]
            self.assertEquals(call_args, VduInstantiateWorkflow.__name__)

    async def test_vnf_instantiate_workflow__activity_change_vnf_instantiation_state_failed__raise_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state_exception,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_instantiate_workflow__activity_change_vnf_instantiation_state_failed__vnf_state_not_updated(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state_exception,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError):
                await self.execute_workflow(self.wf_input)
        self.mock_change_vnf_state_tracker.assert_not_called()

    async def test_vnf_instantiate_workflow__change_vnf_state_failed__raise_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state_exception,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_instantiate_workflow__get_task_queue_failed__raises_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue_raise_exception,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_instantiate_workflow__wf_vnf_prepare_failed__raises_child_wf_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue,
            [
                VnfInstantiateWorkflowImpl,
                MockPrepareVnfWorkflowFailed,
                MockVduInstantiateWorkflow,
            ],
            activities,
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause.cause, ChildWorkflowError))

    async def test_vnf_instantiate_workflow__wf_vdu_instantiate_failed__raises_child_wf_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue,
            [
                VnfInstantiateWorkflowImpl,
                MockPrepareVnfWorkflow,
                MockVduInstantiateWorkflowFailed,
            ],
            activities,
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ChildWorkflowError))

    async def test_vnf_instantiate_workflow__get_vnf_descriptor_failed__raises_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            mock_get_vnf_descriptor_raise_exception,
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_instantiate_workflow__get_vnf_record_failed__raises_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            mock_get_vnf_record_raise_exception,
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_vnf_instantiate_workflow__get_vim_cloud_failed__raises_activity_error(
        self,
    ):
        activities = [
            mock_get_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud_raise_exception,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    @patch("osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl.instantiate_vdus")
    async def test_vnf_instantiate_workflow__use_different_task_queue__use_juju_task_queue(
        self, mock_instantiate_vdus
    ):
        workflows = [VnfInstantiateWorkflowImpl, MockPrepareVnfWorkflow]
        activities = [
            mock_get_different_task_queue,
            self.mock_change_vnf_instantiation_state,
            self.mock_change_vnf_state,
            self.mock_send_notification_for_vnf,
            mock_set_vnf_model,
            MockGetVnfDescriptor(),
            MockGetVnfRecord(),
            mock_get_vim_cloud,
        ]
        async with self.env, self.get_worker(
            self.task_queue, workflows, activities
        ), self.get_worker(juju_task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        mock_instantiate_vdus.assert_called_once_with(
            vnfr=sample_vnfr,
            vnfd=sample_vnfd,
            task_queue=juju_task_queue,
            cloud=cloud,
            vnf_instantiation_config={},
        )


class TestInstantiateVdus(asynctest.TestCase):
    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch(
        "temporalio.workflow.execute_child_workflow",
    )
    async def test_instantiate_vdus__empty_vnf_instantiation_config__child_wf_executed_with_expected_vdu_instantiate_input(
        self,
        mock_execute_child_workflow,
        mock_get_vdu_instantiate_input,
    ):
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input,
            sample_vdu_instantiate_wf_id_1,
        )
        await VnfInstantiateWorkflowImpl.instantiate_vdus(
            sample_vnfr, sample_vnfd, LCM_TASK_QUEUE, cloud, {}
        )
        call_mock_execute_child_workflow = mock_execute_child_workflow.call_args_list
        self.assertEqual(
            call_mock_execute_child_workflow[0].kwargs,
            {
                "arg": sample_vdu_instantiate_input,
                "id": sample_vdu_instantiate_wf_id_1,
                "task_queue": LCM_TASK_QUEUE,
                "workflow": f"{VduInstantiateWorkflow.__name__}",
            },
        )

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch("temporalio.workflow.execute_child_workflow")
    async def test_instantiate_vdus__vnf_instantiation_config_exists__child_wf_executed_with_expected_vdu_instantiate_input(
        self,
        mock_execute_child_workflow,
        mock_get_vdu_instantiate_input,
    ):
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input_with_config,
            sample_vdu_instantiate_wf_id_1,
        )
        await VnfInstantiateWorkflowImpl.instantiate_vdus(
            sample_vnfr, sample_vnfd, LCM_TASK_QUEUE, cloud, vnf_config
        )
        call_mock_execute_child_workflow = mock_execute_child_workflow.call_args_list
        self.assertEqual(
            call_mock_execute_child_workflow[0].kwargs,
            {
                "arg": sample_vdu_instantiate_input_with_config,
                "id": sample_vdu_instantiate_wf_id_1,
                "task_queue": LCM_TASK_QUEUE,
                "workflow": f"{VduInstantiateWorkflow.__name__}",
            },
        )

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch("temporalio.workflow.execute_child_workflow")
    async def test_instantiate_vdus__vnfd_without_vdu__child_wf_is_not_executed(
        self,
        mock_execute_child_workflow,
        mock_get_vdu_instantiate_input,
    ):
        vnfd = deepcopy(sample_vnfd)
        vnfd["vdu"] = []
        await VnfInstantiateWorkflowImpl.instantiate_vdus(
            sample_vnfr, vnfd, LCM_TASK_QUEUE, cloud, {}
        )
        mock_execute_child_workflow.assert_not_called()

    @patch(
        "osm_lcm.temporal.vnf_workflows.VnfInstantiateWorkflowImpl._get_vdu_instantiate_input"
    )
    @patch("temporalio.workflow.execute_child_workflow")
    async def test_instantiate_vdus__use_juju_task_queue__child_wf_executed_with_juju_task_queue(
        self,
        mock_execute_child_workflow,
        mock_get_vdu_instantiate_input,
    ):
        mock_get_vdu_instantiate_input.return_value = (
            sample_vdu_instantiate_input_with_config,
            sample_vdu_instantiate_wf_id_1,
        )
        await VnfInstantiateWorkflowImpl.instantiate_vdus(
            sample_vnfr, sample_vnfd, juju_task_queue, cloud, vnf_config
        )
        call_mock_execute_child_workflow = mock_execute_child_workflow.call_args_list
        self.assertEqual(
            call_mock_execute_child_workflow[0].kwargs,
            {
                "arg": sample_vdu_instantiate_input_with_config,
                "id": sample_vdu_instantiate_wf_id_1,
                "task_queue": juju_task_queue,
                "workflow": f"{VduInstantiateWorkflow.__name__}",
            },
        )


class TestGetVduInstantiateInfo(asynctest.TestCase):
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__empty_vnf_instantiation_config__get_expected_vdu_inst_input_and_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.return_value = {}
        mock_get_application_config.return_value = {}
        (
            vdu_instantiate_input,
            vdu_instantiate_workflow_id,
        ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
            vnfr=sample_vnfr,
            vnfd=sample_vnfd,
            vdu=vdu,
            cloud=cloud,
            vnf_instantiation_config={},
        )
        self.assertTupleEqual(
            (vdu_instantiate_input, vdu_instantiate_workflow_id),
            (sample_vdu_instantiate_input, "my-model-name-my-app"),
        )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__with_vnf_instantiation_config__get_expected_vdu_inst_input_and_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.return_value = config
        mock_get_application_config.return_value = app_config
        (
            vdu_instantiate_input,
            vdu_instantiate_workflow_id,
        ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
            vnfr=sample_vnfr,
            vnfd=sample_vnfd,
            vdu=vdu,
            cloud=cloud,
            vnf_instantiation_config=vnf_config,
        )
        self.assertTupleEqual(
            (vdu_instantiate_input, vdu_instantiate_workflow_id),
            (sample_vdu_instantiate_input_with_config, "my-model-name-my-app"),
        )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__vnfr_without_namespace__raise_type_error(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        vnfr = deepcopy(sample_vnfr)
        vnfr.pop("namespace")
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        with self.assertRaises(TypeError):
            VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__app_name_is_empty__expected_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        charm_info = CharmInfo(app_name="", channel="latest", entity_url="my-url")
        mock_get_application_config.return_value = app_config
        mock_get_charm_info.return_value = charm_info
        (
            _,
            vdu_instantiate_workflow_id,
        ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
            vnfr=sample_vnfr,
            vnfd=sample_vnfd,
            vdu=vdu,
            cloud=cloud,
            vnf_instantiation_config={},
        )
        self.assertEqual(vdu_instantiate_workflow_id, "my-model-name-")

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_compute_constraints_failed__no_vdu_inst_wf_input_and_vdu_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.side_effect = GetComputeConstraintsException(
            "get_compute_constraints failed"
        )
        mock_get_charm_info.return_value = sample_charm_info
        with self.assertRaises(Exception):
            (
                vdu_instantiate_input,
                vdu_instantiate_workflow_id,
            ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )
            self.assertEqual(
                (vdu_instantiate_input, vdu_instantiate_workflow_id), (None, None)
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_compute_constraints_failed__raise_compute_constraints_exception(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.side_effect = GetComputeConstraintsException(
            "get_compute_constraints failed"
        )
        mock_get_charm_info.return_value = sample_charm_info
        with self.assertRaises(GetComputeConstraintsException):
            VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_charm_info_failed__raise_get_charm_info_exception(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.side_effect = GetCharmInfoException(
            "get_compute_constraints failed"
        )
        with self.assertRaises(GetCharmInfoException):
            VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_charm_info_failed__no_vdu_inst_wf_input_and_vdu_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.side_effect = GetCharmInfoException(
            "get_compute_constraints failed"
        )
        with self.assertRaises(Exception):
            (
                vdu_instantiate_input,
                vdu_instantiate_workflow_id,
            ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )
            self.assertEqual(
                (vdu_instantiate_input, vdu_instantiate_workflow_id), (None, None)
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_vdu_instantiation_params_failed__raise_get_vdu_instantiation_params_exception(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.side_effect = (
            GetVduInstantiationParamsException("get_vdu_instantiation_params failed")
        )
        mock_get_application_config.return_value = {}
        with self.assertRaises(GetVduInstantiationParamsException):
            VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_vdu_instantiation_params_failed__no_vdu_inst_wf_input_and_vdu_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.side_effect = (
            GetVduInstantiationParamsException("get_vdu_instantiation_params failed")
        )
        mock_get_application_config.return_value = {}
        with self.assertRaises(Exception):
            (
                vdu_instantiate_input,
                vdu_instantiate_workflow_id,
            ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

            self.assertEqual(
                (vdu_instantiate_input, vdu_instantiate_workflow_id), (None, None)
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_application_config_failed__raise_get_application_config_exception(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.return_value = config
        mock_get_application_config.side_effect = GetApplicationConfigException(
            "get_vdu_instantiation_params failed"
        )
        with self.assertRaises(GetApplicationConfigException):
            VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )

    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_application_config")
    @patch(
        "osm_lcm.temporal.vnf_activities.VnfSpecifications.get_vdu_instantiation_params"
    )
    @patch("osm_lcm.temporal.juju_paas_activities.CharmInfoUtils.get_charm_info")
    @patch("osm_lcm.temporal.vnf_activities.VnfSpecifications.get_compute_constraints")
    async def test_get_vdu_instantiate_input__get_application_config_failed__no_vdu_inst_wf_input_and_vdu_wf_id(
        self,
        mock_get_compute_constraints,
        mock_get_charm_info,
        mock_get_vdu_instantiation_params,
        mock_get_application_config,
    ):
        mock_get_compute_constraints.return_value = sample_constraints
        mock_get_charm_info.return_value = sample_charm_info
        mock_get_vdu_instantiation_params.return_value = config
        mock_get_application_config.side_effect = GetApplicationConfigException(
            "get_vdu_instantiation_params failed"
        )
        with self.assertRaises(Exception):
            (
                vdu_instantiate_input,
                vdu_instantiate_workflow_id,
            ) = VnfInstantiateWorkflowImpl._get_vdu_instantiate_input(
                vnfr=sample_vnfr,
                vnfd=sample_vnfd,
                vdu=vdu,
                cloud=cloud,
                vnf_instantiation_config={},
            )
            self.assertEqual(
                (vdu_instantiate_input, vdu_instantiate_workflow_id), (None, None)
            )


class TestVnfTerminateWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client

        # Maps workflow base classes to implementation classes.
        # Can be overridden by individual tests to swap out implementations by
        # setting self.workflows[<workflow_base_class>] = <implementation_class>
        self.workflows = {
            VduTerminateWorkflow: MockVduTerminateWorkflow,
            VnfTerminateWorkflow: VnfTerminateWorkflowImpl,
        }
        self.mock_get_vnf_record = MockGetVnfRecord()
        self.mock_get_vnf_descriptor = MockGetVnfDescriptor()
        self.mock_change_vnf_instantiation_state = MockChangeVnfInstantiationState()
        self.activities = [
            self.mock_get_vnf_record,
            self.mock_get_vnf_descriptor,
            self.mock_change_vnf_instantiation_state,
        ]
        self.task_queue = LCM_TASK_QUEUE
        self.workflow_input = VnfTerminateWorkflow.Input(
            vnfr_uuid="my_vnfr_uuid",
        )

    def get_worker(self, task_queue: str, activities: list) -> Worker:
        return Worker(
            self.client,
            task_queue=task_queue,
            workflows=list(self.workflows.values()),
            activities=activities,
            debug_mode=DEBUG_MODE,
        )

    async def execute_workflow(self, wf_input: VnfTerminateWorkflow.Input):
        return await self.client.execute_workflow(
            VnfTerminateWorkflowImpl.run,
            arg=wf_input,
            id=uuid.uuid4().hex,
            task_queue=self.task_queue,
            task_timeout=TASK_TIMEOUT,
            execution_timeout=EXECUTION_TIMEOUT,
        )

    @patch("temporalio.workflow.execute_child_workflow")
    async def test_vnf_terminate_workflow__all_activities_called(
        self, mock_execute_child_workflow: CoroutineMock
    ):
        """Test all the expected activities and child workflows are called"""
        async with self.env, self.get_worker(self.task_queue, self.activities):
            await self.execute_workflow(self.workflow_input)

        self.mock_get_vnf_record.tracker.assert_called_once()
        self.mock_get_vnf_descriptor.tracker.assert_called_once()
        mock_execute_child_workflow.assert_called_once()
        self.mock_change_vnf_instantiation_state.tracker.assert_called_once()

    async def test_vnf_terminate_workflow__not_instantiated__logs_warning(self):
        """Test warning is logged when VNF is not instantiated"""
        self.mock_get_vnf_record.return_value = GetVnfRecord.Output(
            vnfr={
                "instantiationState": "NOT_INSTANTIATED",
                "vnfd-id": "test-vnfd-id",
                "vim-account-id": vim_account_id,
                "namespace": model_name,
            }
        )
        async with self.env, self.get_worker(self.task_queue, self.activities):
            with self.assertLogs(level="INFO") as logs:
                await self.execute_workflow(self.workflow_input)

        any(
            "VNF(`my_vnfr_uuid`) is not instantiated" in record.message
            and record.levelname == "WARNING"
            for record in logs.records
        )

    @patch("temporalio.workflow.execute_child_workflow")
    async def test_vnf_terminate_workflow__no_vdu__successful(
        self, mock_execute_child_workflow: CoroutineMock
    ):
        self.mock_get_vnf_descriptor.return_value = GetVnfDescriptor.Output(
            vnfd={
                "id": "test-vnfd-id",
                "vdu": [],
            }
        )

        async with self.env, self.get_worker(self.task_queue, self.activities):
            await self.execute_workflow(self.workflow_input)

        mock_execute_child_workflow.assert_not_called()

    async def test_vnf_terminate_workflow__activity_error__is_raised(self):
        """Validate that activity errors bubble up through the workflow"""
        self.mock_get_vnf_record.side_effect = TestException("get_vnf_record failed")

        async with self.env, self.get_worker(self.task_queue, self.activities):
            with self.assertRaises(WorkflowFailureError):
                await self.execute_workflow(self.workflow_input)

    async def test_vnf_terminate_workflow__child_workflow_error__is_raised(self):
        """Validates that child workflow errors bubble up to the parent workflow

        This might seem unnecessary, but depending on how the co-routines are
        awaited, errors might be silently swallowed.
        """
        self.workflows[VduTerminateWorkflow] = MockVduTerminateWorkflowRaisesException
        async with self.env, self.get_worker(self.task_queue, self.activities):
            with self.assertRaises(WorkflowFailureError):
                await self.execute_workflow(self.workflow_input)

    @parameterized.expand(["vnfd-id", "vim-account-id", "namespace"])
    async def test_vnf_terminate_workflow__no_vnfd_id__raises_error(
        self, vnfr_required_value
    ):
        """Test that a missing required value in the VNFR raises an error"""
        vnfr = {
            "instantiationState": "NOT_INSTANTIATED",
            "vnfd-id": "test-vnfd-id",
            "vim-account-id": "test-vim-account-id",
            "namespace": "test-namespace",
        }
        del vnfr[vnfr_required_value]
        self.mock_get_vnf_record.return_value = GetVnfRecord.Output(vnfr=vnfr)

        async with self.env, self.get_worker(self.task_queue, self.activities):
            with self.assertRaises(WorkflowFailureError):
                await self.execute_workflow(self.workflow_input)

    @patch("temporalio.workflow.execute_child_workflow")
    async def test_vnf_terminate_workflow__no_vdu_id__is_ignored(
        self, mock_execute_child_workflow: CoroutineMock
    ):
        """Test that VDUs without an ID are ignored

        This is not a business requirement, but a defensive measure to avoid
        errors in case the VNFD is malformed."""
        self.mock_get_vnf_descriptor.return_value = GetVnfDescriptor.Output(
            vnfd={
                "id": "test-vnfd-id",
                "vdu": [
                    {
                        "id": "test-vdu-id",
                    },
                    {
                        "not-id": "test-not-id",  # No ID in this VDU, it will be ignored
                    },
                ],
            }
        )

        async with self.env, self.get_worker(self.task_queue, self.activities):
            await self.execute_workflow(self.workflow_input)

        mock_execute_child_workflow.assert_called_once()
        self.assertEquals(
            mock_execute_child_workflow.call_args.kwargs["arg"].application_name,
            "test-vdu-id",
        )


class TestVnfDeleteWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client
        self.task_queue = LCM_TASK_QUEUE
        self.wf_input = VnfDeleteWorkflowImpl.Input(vnfr_uuid=vnfr_uuid)
        self.mock_delete_vnf_record_tracker = Mock()

    def assert_vnf_record_is_called(self):
        self.assertEqual(
            self.mock_delete_vnf_record_tracker.call_args_list[0].args[0],
            DeleteVnfRecord.Input(vnfr_uuid=vnfr_uuid),
        )

    @activity.defn(name=DeleteVnfRecord.__name__)
    async def mock_delete_vnf_record(
        self,
        delete_vnf_record_input: DeleteVnfRecord.Input,
    ) -> None:
        self.mock_delete_vnf_record_tracker(delete_vnf_record_input)

    @activity.defn(name=DeleteVnfRecord.__name__)
    async def mock_delete_vnf_record_raises_exception(
        self,
        delete_vnf_record_input: DeleteVnfRecord.Input,
    ) -> None:
        self.mock_delete_vnf_record_tracker(delete_vnf_record_input)
        raise TestException(f"{DeleteVnfRecord.__name__} failed.")

    async def test_vnf_delete_workflow__successful(self):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfDeleteWorkflowImpl],
                activities=[
                    mock_get_vnf_record_not_instantiated_state,
                    self.mock_delete_vnf_record,
                ],
            ):
                await self.client.execute_workflow(
                    VnfDeleteWorkflowImpl.run,
                    arg=self.wf_input,
                    id=f"{VnfDeleteWorkflowImpl.__name__}-{self.wf_input.vnfr_uuid}",
                    task_queue=self.task_queue,
                    task_timeout=TASK_TIMEOUT,
                    execution_timeout=EXECUTION_TIMEOUT,
                )
                self.assert_vnf_record_is_called()

    async def test_vnf_delete_workflow__instantiated_state__fails(self):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfDeleteWorkflowImpl],
                activities=[mock_get_vnf_record_instantiated_state],
            ):
                with self.assertRaises(WorkflowFailureError) as err:
                    await self.client.execute_workflow(
                        VnfDeleteWorkflowImpl.run,
                        arg=self.wf_input,
                        id=f"{VnfDeleteWorkflowImpl.__name__}-{self.wf_input.vnfr_uuid}",
                        task_queue=self.task_queue,
                        task_timeout=TASK_TIMEOUT,
                        execution_timeout=EXECUTION_TIMEOUT,
                    )
                self.assertIsInstance(err.exception.cause, ApplicationError)
                self.assertEqual(
                    str(err.exception.cause),
                    "VNF must be in instantiation state 'NOT_INSTANTIATED'",
                )
                self.mock_delete_vnf_record_tracker.assert_not_called()

    async def test_vnf_delete_workflow__get_vnf_record_fails__workflow_fails(self):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfDeleteWorkflowImpl],
                activities=[mock_get_vnf_record_raise_exception],
            ):
                with self.assertRaises(WorkflowFailureError) as err:
                    await self.client.execute_workflow(
                        VnfDeleteWorkflowImpl.run,
                        arg=self.wf_input,
                        id=f"{VnfDeleteWorkflowImpl.__name__}-{self.wf_input.vnfr_uuid}",
                        task_queue=self.task_queue,
                        task_timeout=TASK_TIMEOUT,
                        execution_timeout=EXECUTION_TIMEOUT,
                    )
                self.assertIsInstance(err.exception.cause, ActivityError)
                self.assertEqual(
                    str(err.exception.cause.cause),
                    f"TestException: {GetVnfRecord.__name__} failed.",
                )
                self.mock_delete_vnf_record_tracker.assert_not_called()

    async def test_vnf_delete_workflow__delete_vnf_record_fails__workflow_fails(self):
        async with self.env:
            async with Worker(
                self.client,
                debug_mode=DEBUG_MODE,
                task_queue=self.task_queue,
                workflows=[VnfDeleteWorkflowImpl],
                activities=[
                    mock_get_vnf_record_not_instantiated_state,
                    self.mock_delete_vnf_record_raises_exception,
                ],
            ):
                with self.assertRaises(WorkflowFailureError) as err:
                    await self.client.execute_workflow(
                        VnfDeleteWorkflowImpl.run,
                        arg=self.wf_input,
                        id=f"{VnfDeleteWorkflowImpl.__name__}-{self.wf_input.vnfr_uuid}",
                        task_queue=self.task_queue,
                        task_timeout=TASK_TIMEOUT,
                        execution_timeout=EXECUTION_TIMEOUT,
                    )
                self.assertIsInstance(err.exception.cause, ActivityError)
                self.assertEqual(
                    str(err.exception.cause.cause),
                    f"TestException: {DeleteVnfRecord.__name__} failed.",
                )
                self.assert_vnf_record_is_called()


if __name__ == "__main__":
    asynctest.main()
