#######################################################################################
# Copyright ETSI Contributors and Others.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import asyncio
import copy
from datetime import timedelta
from typing import Any
from unittest.mock import Mock, patch
import uuid

import asynctest
from temporalio import activity, workflow
from temporalio.client import WorkflowFailureError
from temporalio.common import RetryPolicy
from temporalio.exceptions import (
    ActivityError,
    ApplicationError,
    ChildWorkflowError,
    RetryState,
    TimeoutError,
)
from temporalio.testing import WorkflowEnvironment
from temporalio.worker import Worker

from osm_common.temporal.activities.lcm import (
    UpdateNsLcmOperationState,
)
from osm_common.temporal.activities.ns import (
    GetNsRecord,
    GetVnfDetails,
    UpdateNsState,
    DeleteNsRecord,
)
from osm_common.temporal.activities.paas import (
    CreateModel,
    RemoveModel,
    CheckModelIsRemoved,
)
from osm_common.temporal.activities.vnf import GetModelNames
from osm_common.temporal.states import LcmOperationState, NsState
from osm_common.temporal.workflows.lcm import LcmOperationWorkflow
from osm_common.temporal.workflows.vnf import (
    VnfInstantiateWorkflow,
    VnfTerminateWorkflow,
    VnfDeleteWorkflow,
)
from osm_common.temporal_task_queues.task_queues_mappings import LCM_TASK_QUEUE
from osm_lcm.temporal.ns_workflows import (
    NsDeleteRecordsWorkflow,
    NsDeleteRecordsWorkflowImpl,
    NsInstantiateWorkflow,
    NsInstantiateWorkflowImpl,
    NsTerminateWorkflow,
    NsTerminateWorkflowImpl,
)

vnfr_uuid_1 = "828d91ee-fa04-43bb-8471-f66ea74597e7"
vnfr_uuid_2 = "c4bbeb41-df7e-4daa-863d-c8fd29fac96d"
vnf_member_index_ref_1 = "vnf-profile-id"
vnf_member_index_ref_2 = "vnf-profile-id-2"
vnf_details = [
    (vnfr_uuid_1, vnf_member_index_ref_1),
    (vnfr_uuid_2, vnf_member_index_ref_2),
]
ns_uuid = "9c96e9c1-aae2-4c61-a85f-2ec8acb448fc"
vim_uuid = "a64f7c6c-bc27-4ec8-b664-5500a3324eca"
vdu_id = "test-vdu"
model_name = "2ec8acb448fc-5500a3324eca"
vnf_config = {
    "member-vnf-index": vnf_member_index_ref_1,
    "vdu": [
        {
            "id": vdu_id,
            "configurable-properties": {
                "config::redirect-map": "https://osm.instantiation.params"
            },
        }
    ],
}
sample_nsr = {
    "_id": ns_uuid,
    "name": "sol006_juju23",
    "name-ref": "sol006_juju23",
    "short-name": "sol006_juju23",
    "admin-status": "ENABLED",
    "nsState": "NOT_INSTANTIATED",
    "instantiate_params": {
        "nsdId": ns_uuid,
        "nsName": "sol006_juju23",
        "nsDescription": "default description",
        "vimAccountId": vim_uuid,
        "vnf": [vnf_config],
    },
}
mock_get_namespace = Mock()
retry_policy = RetryPolicy(
    initial_interval=timedelta(seconds=1),
    backoff_coefficient=2.0,
    maximum_interval=None,
    maximum_attempts=1,
    non_retryable_error_types=None,
)
SANDBOXED = False
DEBUG_MODE = True
TASK_TIMEOUT = timedelta(seconds=5)
EXECUTION_TIMEOUT = timedelta(seconds=10)


class TestException(Exception):
    pass


class MockActivity:
    def __init__(self, *args: Any, **kwargs: Any):
        self.tracker = Mock(return_value=None)

    async def __call__(self, *args: Any, **kwargs: Any) -> Any:
        return self.tracker(*args, **kwargs)

    def __setattr__(self, __name: str, __value: Any) -> None:
        if __name in ("return_value", "side_effect"):
            self.tracker.__setattr__(__name, __value)
        else:
            super().__setattr__(__name, __value)


@activity.defn(name=CreateModel.__name__)
async def mock_create_model(create_model_input: CreateModel.Input) -> None:
    pass


@activity.defn(name=CreateModel.__name__)
async def mock_create_model_raises(create_model_input: CreateModel.Input) -> None:
    raise TestException(f"{CreateModel.__name__} failed.")


@activity.defn(name=RemoveModel.__name__)
async def mock_remove_model(remove_model_input: RemoveModel.Input) -> None:
    pass


@activity.defn(name=RemoveModel.__name__)
async def mock_remove_model_raises(remove_model_input: RemoveModel.Input) -> None:
    raise TestException(f"{RemoveModel.__name__} failed.")


@activity.defn(name=CheckModelIsRemoved.__name__)
async def mock_check_model_is_removed(
    check_model_is_removed_input: CheckModelIsRemoved.Input,
) -> None:
    pass


@activity.defn(name=CheckModelIsRemoved.__name__)
async def mock_check_model_is_removed_raises(
    check_model_is_removed_input: CheckModelIsRemoved.Input,
) -> None:
    raise TestException(f"{CheckModelIsRemoved.__name__} failed.")


@activity.defn(name=CheckModelIsRemoved.__name__)
async def mock_check_model_is_removed_cancelled(
    check_model_is_removed_input: CheckModelIsRemoved.Input,
) -> None:
    raise asyncio.exceptions.CancelledError(
        f"{CheckModelIsRemoved.__name__} cancelled."
    )


@activity.defn(name=GetModelNames.__name__)
async def mock_get_model_names(
    get_model_names_input: GetModelNames.Input,
) -> GetModelNames.Output:
    return GetModelNames.Output({"namespace-1", "namespace-2"})


@activity.defn(name=GetModelNames.__name__)
async def mock_get_model_names_raises(
    get_model_names_input: GetModelNames.Input,
) -> GetModelNames.Output:
    raise TestException(f"{GetModelNames.__name__} failed.")


@activity.defn(name=GetVnfDetails.__name__)
class MockGetVnfDetails(MockActivity):
    def __init__(self, *args: Any, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self.tracker.return_value = GetVnfDetails.Output(vnf_details=vnf_details)


@activity.defn(name=GetVnfDetails.__name__)
class MockGetVnfDetailsRaises(MockActivity):
    def __init__(self, *args: Any, **kwargs: Any):
        super().__init__(*args, **kwargs)
        self.tracker.side_effect = TestException(f"{GetVnfDetails.__name__} failed.")


@activity.defn(name=GetNsRecord.__name__)
async def mock_get_ns_record(
    get_ns_record_input: GetNsRecord.Input,
) -> GetNsRecord.Output:
    return GetNsRecord.Output(nsr=sample_nsr)


@activity.defn(name=GetNsRecord.__name__)
async def mock_get_ns_record_raise_exception(
    get_ns_record_input: GetNsRecord.Input,
) -> GetNsRecord.Output:
    raise TestException(f"{GetNsRecord.__name__} failed.")


@activity.defn(name=GetNsRecord.__name__)
async def mock_get_ns_record_not_instantiated_state(
    get_ns_record_input: GetNsRecord.Input,
) -> GetNsRecord.Output:
    nsr = copy.deepcopy(sample_nsr)
    nsr["nsState"] = NsState.NOT_INSTANTIATED.name
    return GetNsRecord.Output(nsr=nsr)


@activity.defn(name=GetNsRecord.__name__)
async def mock_get_ns_record_instantiated_state(
    get_ns_record_input: GetNsRecord.Input,
) -> GetNsRecord.Output:
    nsr = copy.deepcopy(sample_nsr)
    nsr["nsState"] = NsState.INSTANTIATED.name
    return GetNsRecord.Output(nsr=nsr)


@workflow.defn(name=VnfInstantiateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfInstantiateWorkflow:
    @workflow.run
    async def run(self, input: VnfInstantiateWorkflow.Input) -> None:
        pass


@workflow.defn(name=VnfInstantiateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfInstantiateWorkflowFailed:
    @workflow.run
    async def run(self, input: VnfInstantiateWorkflow.Input) -> None:
        raise ChildWorkflowError(
            message=f"{VnfInstantiateWorkflow.__name__} child workflow failed.",
            namespace="default",
            workflow_id="123",
            run_id="1",
            workflow_type=VnfInstantiateWorkflow.__name__,
            initiated_event_id=0,
            started_event_id=0,
            retry_state=RetryState.NON_RETRYABLE_FAILURE,
        )


@workflow.defn(name=VnfTerminateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfTerminateWorkflow:
    @workflow.run
    async def run(self, workflow_input: VnfTerminateWorkflow.Input) -> None:
        pass


@workflow.defn(name=VnfTerminateWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfTerminateWorkflowFailed:
    @workflow.run
    async def run(self, workflow_input: VnfTerminateWorkflow.Input) -> None:
        raise ChildWorkflowError(
            message=f"{VnfTerminateWorkflow.__name__} child workflow failed.",
            namespace="default",
            workflow_id="123",
            run_id="1",
            workflow_type=VnfTerminateWorkflow.__name__,
            initiated_event_id=0,
            started_event_id=0,
            retry_state=RetryState.NON_RETRYABLE_FAILURE,
        )


@workflow.defn(name=VnfDeleteWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfDeleteWorkflow:
    @workflow.run
    async def run(self, input: VnfDeleteWorkflow.Input) -> None:
        pass


@workflow.defn(name=VnfDeleteWorkflow.__name__, sandboxed=SANDBOXED)
class MockVnfDeleteWorkflowFailed:
    @workflow.run
    async def run(self, input: VnfDeleteWorkflow.Input) -> None:
        raise ChildWorkflowError(
            message=f"{VnfDeleteWorkflow.__name__} child workflow failed.",
            namespace="default",
            workflow_id="123",
            run_id="1",
            workflow_type=VnfDeleteWorkflow.__name__,
            initiated_event_id=0,
            started_event_id=0,
            retry_state=RetryState.NON_RETRYABLE_FAILURE,
        )


@patch(
    "osm_lcm.temporal.ns_workflows.NsInstantiateWorkflowImpl._get_namespace",
    new=mock_get_namespace,
)
class TestNsInstantiateWorkflow(asynctest.TestCase):
    input = LcmOperationWorkflow.Input(
        nslcmop={
            "_id": "1234",
            "nsInstanceId": ns_uuid,
            "operationParams": {"vimAccountId": vim_uuid},
        }
    )

    @activity.defn(name=UpdateNsLcmOperationState.__name__)
    async def mock_update_lcm_operation_state(
        self,
        data: UpdateNsLcmOperationState.Input,
    ) -> None:
        self.mock_update_lcm_operation_state_tracker(data)

    @activity.defn(name=UpdateNsState.__name__)
    async def mock_update_ns_state(self, data: UpdateNsState.Input) -> None:
        self.mock_update_ns_state_tracker(data)

    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client
        self.mock_update_lcm_operation_state_tracker = Mock()
        self.mock_update_ns_state_tracker = Mock()
        self.workflows = [NsInstantiateWorkflowImpl, MockVnfInstantiateWorkflow]
        self.task_queue = LCM_TASK_QUEUE
        mock_get_namespace.return_value = model_name

    def get_worker(self, task_queue: str, workflows: list, activities: list) -> Worker:
        return Worker(
            self.client,
            task_queue=task_queue,
            workflows=workflows,
            activities=activities,
            debug_mode=DEBUG_MODE,
        )

    async def execute_workflow(self):
        return await self.client.execute_workflow(
            NsInstantiateWorkflow.__name__,
            arg=self.input,
            id=self.input.nslcmop["nsInstanceId"],
            task_queue=self.task_queue,
            task_timeout=TASK_TIMEOUT,
            execution_timeout=EXECUTION_TIMEOUT,
        )

    async def test_instantiate_workflow__successful__update_lcm_op_state(self):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            await self.execute_workflow()
        self.assert_lcm_op_states(
            self.mock_update_lcm_operation_state_tracker.call_args_list,
            [LcmOperationState.PROCESSING, LcmOperationState.COMPLETED],
        )

    async def test_instantiate_workflow__successful__update_ns_state(self):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            await self.execute_workflow()
        self.assert_ns_states(
            self.mock_update_ns_state_tracker.call_args_list, [NsState.INSTANTIATED]
        )

    async def test_instantiate_workflow__activity_failed__update_ns_state(
        self,
    ):
        activities = [
            mock_create_model_raises,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_ns_states(
            self.mock_update_ns_state_tracker.call_args_list, [NsState.INSTANTIATED]
        )

    async def test_instantiate_workflow__activity_failed__update_lcm_op_state(
        self,
    ):
        activities = [
            mock_create_model_raises,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_lcm_op_states(
            self.mock_update_lcm_operation_state_tracker.call_args_list,
            [LcmOperationState.PROCESSING, LcmOperationState.FAILED],
        )

    async def test_instantiate_workflow__create_model_activity_failed__raise_activity_error(
        self,
    ):
        activities = [
            mock_create_model_raises,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError) as err:
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_instantiate_workflow__create_model_activity_failed__error_details_expected(
        self,
    ):
        activities = [
            mock_create_model_raises,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_ns_error_details(
            self.mock_update_ns_state_tracker.call_args_list,
            [f"TestException: {CreateModel.__name__} failed."],
        )

    async def test_instantiate_workflow__get_ns_record_activity_failed__error_details_expected(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record_raise_exception,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_ns_error_details(
            self.mock_update_ns_state_tracker.call_args_list,
            [f"TestException: {GetNsRecord.__name__} failed."],
        )

    async def test_instantiate_workflow__get_ns_record_activity_failed__raise_activity_error(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record_raise_exception,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError) as err:
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_instantiate_workflow__get_vnf_details_activity_failed__error_details_expected(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetailsRaises(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_ns_error_details(
            self.mock_update_ns_state_tracker.call_args_list,
            [f"TestException: {GetVnfDetails.__name__} failed."],
        )

    async def test_instantiate_workflow__get_vnf_details_activity_failed__raise_activity_error(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetailsRaises(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError) as err:
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assertTrue(isinstance(err.exception.cause, ActivityError))

    @patch("osm_lcm.temporal.ns_workflows.NsInstantiateWorkflowImpl.get_vnf_config")
    async def test_instantiate_workflow__get_vnf_config_failed__workflow_times_out(
        self, mock_get_vnf_config
    ):
        # Because of workflow task timeout policy, workflow times out
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        mock_get_vnf_config.side_effect = TestException("get_vnf_config failed.")
        with self.assertRaises(WorkflowFailureError) as err:
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assertTrue(isinstance(err.exception.cause, TimeoutError))

    @patch("osm_lcm.temporal.ns_workflows.NsInstantiateWorkflowImpl.get_vnf_config")
    async def test_instantiate_workflow__get_vnf_config_failed__update_ns_state(
        self, mock_get_vnf_config
    ):
        # Workflow task failure
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        mock_get_vnf_config.side_effect = TestException("get_vnf_config failed.")
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_ns_states(
            self.mock_update_ns_state_tracker.call_args_list, [NsState.INSTANTIATED]
        )

    @patch("osm_lcm.temporal.ns_workflows.NsInstantiateWorkflowImpl.get_vnf_config")
    async def test_instantiate_workflow__get_vnf_config_failed__update_lcm_op_state(
        self, mock_get_vnf_config
    ):
        # Workflow task failure
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        mock_get_vnf_config.side_effect = TestException("get_vnf_config failed.")
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue, self.workflows, activities
            ):
                await self.execute_workflow()
        self.assert_lcm_op_states(
            self.mock_update_lcm_operation_state_tracker.call_args_list,
            [LcmOperationState.PROCESSING, LcmOperationState.FAILED],
        )

    async def test_instantiate_workflow__child_wf_failed__raise_child_workflow_error(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError) as err:
            async with self.env, self.get_worker(
                self.task_queue,
                [NsInstantiateWorkflowImpl, MockVnfInstantiateWorkflowFailed],
                activities,
            ):
                await self.execute_workflow()
        self.assertTrue(isinstance(err.exception.cause, ChildWorkflowError))

    async def test_instantiate_workflow__child_wf_failed__error_details_expected(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue,
                [NsInstantiateWorkflowImpl, MockVnfInstantiateWorkflowFailed],
                activities,
            ):
                await self.execute_workflow()
        self.assert_ns_error_details(
            self.mock_update_ns_state_tracker.call_args_list,
            [f"{VnfInstantiateWorkflow.__name__} child workflow failed."],
        )

    async def test_instantiate_workflow__child_wf_failed__update_ns_state(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]
        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue,
                [NsInstantiateWorkflowImpl, MockVnfInstantiateWorkflowFailed],
                activities,
            ):
                await self.execute_workflow()
        self.assert_ns_states(
            self.mock_update_ns_state_tracker.call_args_list, [NsState.INSTANTIATED]
        )

    async def test_instantiate_workflow__child_wf_failed__update_lcm_op_state(
        self,
    ):
        activities = [
            mock_create_model,
            MockGetVnfDetails(),
            mock_get_ns_record,
            self.mock_update_ns_state,
            self.mock_update_lcm_operation_state,
        ]

        with self.assertRaises(WorkflowFailureError):
            async with self.env, self.get_worker(
                self.task_queue,
                [NsInstantiateWorkflowImpl, MockVnfInstantiateWorkflowFailed],
                activities,
            ):
                await self.execute_workflow()
        self.assert_lcm_op_states(
            self.mock_update_lcm_operation_state_tracker.call_args_list,
            [LcmOperationState.PROCESSING, LcmOperationState.FAILED],
        )

    def assert_ns_states(self, call_args_list, expected_states):
        """Asserts that the NS state was set to each of the expected states (in order)."""
        self.assertEqual(
            [call.args[0].state for call in call_args_list], expected_states
        )

    def assert_ns_error_details(self, call_args_list, expected_error):
        """Asserts that the NS state was set to each of the expected states (in order)."""
        self.assertEqual(
            [call.args[0].message for call in call_args_list], expected_error
        )

    def assert_lcm_op_states(self, call_args_list, expected_states):
        """Asserts that the LCM operation was set to each of the expected states (in order)."""
        self.assertEqual(
            [call.args[0].op_state for call in call_args_list], expected_states
        )


class TestGetVnfConfig(asynctest.TestCase):
    def test_vnf_config__nsr_config_matches_with_vnf__return_expected_config(self):
        expected_config = {
            "member-vnf-index": "vnf-profile-id",
            "vdu": [
                {
                    "configurable-properties": {
                        "config::redirect-map": "https://osm.instantiation.params"
                    },
                    "id": "test-vdu",
                }
            ],
        }
        result = NsInstantiateWorkflowImpl.get_vnf_config(
            vnf_member_index_ref_1, sample_nsr
        )
        self.assertEqual(result, expected_config)

    def test_vnf_config__nsr_does_not_have_config__expected_no_config(self):
        nsr = copy.deepcopy(sample_nsr)
        nsr["instantiate_params"] = {}
        result = NsInstantiateWorkflowImpl.get_vnf_config(vnf_member_index_ref_1, nsr)
        self.assertEqual(result, {})

    def test_vnf_config__empty_vnf_member_index_ref__expected_no_config(self):
        result = NsInstantiateWorkflowImpl.get_vnf_config("", sample_nsr)
        self.assertEqual(result, {})

    def test_vnf_config__nsr_config_does_not_match_with_vnf__expected_no_config(self):
        result = NsInstantiateWorkflowImpl.get_vnf_config(
            vnf_member_index_ref_2, sample_nsr
        )
        self.assertEqual(result, {})


class TestNsTerminateWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.wf_input = NsTerminateWorkflow.Input(
            ns_uuid="ns-uuid",
            vim_uuid="vim-uuid",
        )
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client

        # Maps workflow base classes to implementation classes.
        # Can be overridden by individual tests to swap out implementations by
        # setting self.workflows[<workflow_base_class>] = <implementation_class>
        self.workflows = [MockVnfTerminateWorkflow, NsTerminateWorkflowImpl]
        self.mock_get_vnf_details = MockGetVnfDetails()
        self.mock_get_models_names = mock_get_model_names
        self.task_queue = LCM_TASK_QUEUE
        self.workflow_input = NsTerminateWorkflow.Input(
            ns_uuid="my-ns-uuid",
            vim_uuid="my-vim-uuid",
        )

    def get_worker(self, task_queue: str, workflows: list, activities: list) -> Worker:
        return Worker(
            self.client,
            task_queue=task_queue,
            workflows=workflows,
            activities=activities,
            debug_mode=DEBUG_MODE,
        )

    async def execute_workflow(self, wf_input):
        return await self.client.execute_workflow(
            NsTerminateWorkflowImpl.run,
            arg=wf_input,
            id=uuid.uuid4().hex,
            task_queue=self.task_queue,
            task_timeout=TASK_TIMEOUT,
            execution_timeout=EXECUTION_TIMEOUT,
        )

    @patch(
        "temporalio.workflow.execute_child_workflow",
        wraps=workflow.execute_child_workflow,
    )
    async def test_ns_terminate_workflow__successful__executes_child_workflow(
        self,
        mock_execute_child_workflow: asynctest.CoroutineMock,
    ):
        workflows = [
            MockVnfTerminateWorkflow,
            NsTerminateWorkflowImpl,
        ]
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        self.assertTrue(mock_execute_child_workflow.called)
        self.assertEqual(mock_execute_child_workflow.call_count, 2)
        call_args = mock_execute_child_workflow.call_args_list[0].kwargs["workflow"]
        self.assertEqual(call_args, VnfTerminateWorkflow.__name__)

    @patch(
        "temporalio.workflow.execute_child_workflow",
        wraps=workflow.execute_child_workflow,
    )
    async def test_ns_terminate_workflow__successful__executes_child_workflows_concurrently(
        self,
        mock_execute_child_workflow: asynctest.CoroutineMock,
    ):
        # Choose a number of child workflows that would take longer to execute than the execution timeout
        num_child_workflows = EXECUTION_TIMEOUT.seconds * 2 + 1
        workflows = [
            MockVnfTerminateWorkflow,
            NsTerminateWorkflowImpl,
        ]
        mock_get_vnf_details = MockGetVnfDetails()
        mock_get_vnf_details.return_value = GetVnfDetails.Output(
            vnf_details=[
                (f"vnfr-{uuid.uuid4().hex}", vnf_member_index_ref_1)
                for _ in range(num_child_workflows)
            ]
        )
        activities = [
            mock_get_vnf_details,
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed,
        ]

        # Set up the return values with a unique workflow ID for each child workflow
        mock_get_model_names.side_effect = [
            f"namespace-{uuid.uuid4().hex}" for _ in range(num_child_workflows)
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)

        self.assertEqual(mock_execute_child_workflow.call_count, num_child_workflows)
        # Check that VnfTerminateWorkflow is executed
        for i in range(0, num_child_workflows):
            call_args = mock_execute_child_workflow.call_args_list[i].kwargs["workflow"]
            self.assertEqual(call_args, VnfTerminateWorkflow.__name__)

    async def test_ns_terminate_workflow__get_vnf_details_failed__raises_activity_error(
        self,
    ):
        activities = [
            MockGetVnfDetailsRaises(),
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_ns_terminate_workflow__get_model_names_failed__raises_activity_error(
        self,
    ):
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names_raises,
            mock_remove_model,
            mock_check_model_is_removed,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_ns_terminate_workflow__remove_model_failed__raises_activity_error(
        self,
    ):
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names,
            mock_remove_model_raises,
            mock_check_model_is_removed,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_ns_terminate_workflow__check_model_removed_failed__raises_activity_error(
        self,
    ):
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed_raises,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))

    async def test_ns_terminate_workflow__wf_vnf_terminate_failed__raises_child_wf_error(
        self,
    ):
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed,
        ]
        async with self.env, self.get_worker(
            self.task_queue,
            [
                NsTerminateWorkflowImpl,
                MockVnfTerminateWorkflowFailed,
            ],
            activities,
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause.cause, ChildWorkflowError))

    async def test_ns_terminate__check_model_is_removed_cancelled__activity_error_is_raised(
        self,
    ):
        activities = [
            MockGetVnfDetails(),
            mock_get_model_names,
            mock_remove_model,
            mock_check_model_is_removed_cancelled,
        ]
        async with self.env, self.get_worker(
            self.task_queue, self.workflows, activities
        ):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertTrue(isinstance(err.exception.cause, ActivityError))


class TestNsDeleteRecordsWorkflow(asynctest.TestCase):
    async def setUp(self):
        self.env = await WorkflowEnvironment.start_time_skipping()
        self.client = self.env.client
        self.task_queue = LCM_TASK_QUEUE
        self.wf_input = NsDeleteRecordsWorkflow.Input(ns_uuid=ns_uuid)
        self.mock_delete_ns_record_tracker = Mock()

    def get_worker(self, task_queue: str, workflows: list, activities: list) -> Worker:
        return Worker(
            self.client,
            task_queue=task_queue,
            workflows=workflows,
            activities=activities,
            debug_mode=DEBUG_MODE,
        )

    async def execute_workflow(self, wf_input):
        return await self.client.execute_workflow(
            NsDeleteRecordsWorkflowImpl.run,
            arg=wf_input,
            id=uuid.uuid4().hex,
            task_queue=self.task_queue,
            task_timeout=TASK_TIMEOUT,
            execution_timeout=EXECUTION_TIMEOUT,
        )

    def assert_ns_record_is_called(self):
        self.assertEqual(
            self.mock_delete_ns_record_tracker.call_args_list[0].args[0],
            DeleteNsRecord.Input(ns_uuid=ns_uuid),
        )

    @activity.defn(name=DeleteNsRecord.__name__)
    async def mock_delete_ns_record(
        self,
        delete_ns_record_input: DeleteNsRecord.Input,
    ) -> None:
        self.mock_delete_ns_record_tracker(delete_ns_record_input)

    @activity.defn(name=DeleteNsRecord.__name__)
    async def mock_delete_ns_record_raises_exception(
        self,
        delete_ns_record_input: DeleteNsRecord.Input,
    ) -> None:
        self.mock_delete_ns_record_tracker(delete_ns_record_input)
        raise TestException(f"{DeleteNsRecord.__name__} failed.")

    async def test_delete_ns_records_workflow__successful(self):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            MockGetVnfDetails(),
            mock_get_ns_record_not_instantiated_state,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
            self.assert_ns_record_is_called()

    @patch(
        "temporalio.workflow.execute_child_workflow",
        wraps=workflow.execute_child_workflow,
    )
    async def test_ns_terminate_workflow__successful__executes_vnf_delete_child_workflows_concurrently(
        self,
        mock_execute_child_workflow: asynctest.CoroutineMock,
    ):
        # Choose a number of child workflows that would take longer to execute than the execution timeout
        num_child_workflows = EXECUTION_TIMEOUT.seconds * 2 + 1
        mock_get_vnf_details = MockGetVnfDetails()
        mock_get_vnf_details.return_value = GetVnfDetails.Output(
            vnf_details=[
                (f"vnfr-{uuid.uuid4().hex}", vnf_member_index_ref_1)
                for _ in range(num_child_workflows)
            ]
        )
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            mock_get_vnf_details,
            mock_get_ns_record_not_instantiated_state,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            await self.execute_workflow(self.wf_input)
        self.assert_ns_record_is_called()
        self.assertEqual(mock_execute_child_workflow.call_count, num_child_workflows)
        # Check that VnfDeleteWorkflow is executed
        for i in range(0, num_child_workflows):
            call_args = mock_execute_child_workflow.call_args_list[i].kwargs["workflow"]
            self.assertEqual(call_args, VnfDeleteWorkflow.__name__)

    async def test_delete_ns_records_workflow__instantiated_state__fails(self):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            MockGetVnfDetails(),
            mock_get_ns_record_instantiated_state,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertIsInstance(err.exception.cause, ApplicationError)
            self.assertEqual(
                str(err.exception.cause),
                "NS must be in NOT_INSTANTIATED state",
            )
            self.mock_delete_ns_record_tracker.assert_not_called()

    async def test_delete_ns_records_workflow__get_vnf_details_fails__workflow_fails(
        self,
    ):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            MockGetVnfDetailsRaises(),
            mock_get_ns_record_not_instantiated_state,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertIsInstance(err.exception.cause, ActivityError)
            self.assertEqual(
                str(err.exception.cause.cause),
                f"TestException: {GetVnfDetails.__name__} failed.",
            )
            self.mock_delete_ns_record_tracker.assert_not_called()

    async def test_delete_ns_records_workflow__get_ns_record_fails__workflow_fails(
        self,
    ):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            MockGetVnfDetails(),
            mock_get_ns_record_raise_exception,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertIsInstance(err.exception.cause, ActivityError)
            self.assertEqual(
                str(err.exception.cause.cause),
                f"TestException: {GetNsRecord.__name__} failed.",
            )
            self.mock_delete_ns_record_tracker.assert_not_called()

    async def test_delete_ns_records_workflow__delete_ns_record_fails__workflow_fails(
        self,
    ):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflow]
        activities = [
            MockGetVnfDetails(),
            mock_get_ns_record_not_instantiated_state,
            self.mock_delete_ns_record_raises_exception,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertIsInstance(err.exception.cause, ActivityError)
            self.assertEqual(
                str(err.exception.cause.cause),
                f"TestException: {DeleteNsRecord.__name__} failed.",
            )
            self.mock_delete_ns_record_tracker.assert_called()

    async def test_delete_ns_records_workflow__vnf_delete_workflow_failed__workflow_fails(
        self,
    ):
        workflows = [NsDeleteRecordsWorkflowImpl, MockVnfDeleteWorkflowFailed]
        activities = [
            MockGetVnfDetails(),
            mock_get_ns_record_not_instantiated_state,
            self.mock_delete_ns_record,
        ]
        async with self.env, self.get_worker(self.task_queue, workflows, activities):
            with self.assertRaises(WorkflowFailureError) as err:
                await self.execute_workflow(self.wf_input)
            self.assertIsInstance(err.exception.cause, ChildWorkflowError)
            self.assertEqual(
                str(err.exception.cause.cause),
                "VnfDeleteWorkflow child workflow failed.",
            )
            self.mock_delete_ns_record_tracker.assert_not_called()
