# Copyright ETSI Contributors and Others.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
#     Unless required by applicable law or agreed to in writing, software
#     distributed under the License is distributed on an "AS IS" BASIS,
#     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#     See the License for the specific language governing permissions and
#     limitations under the License.

from unittest import TestCase
from osm_lcm.temporal.juju_paas_activities import CharmInfoUtils
from osm_common.temporal.dataclasses_common import CharmInfo
import yaml

vdu_nominal = """
---
vdu:
   - id: test-vdu-id
     name: test-vdu-name
     int-cpd:
       - id: internal
         int-virtual-link-desc: network1
       - id: mgmt
     virtual-compute-desc: compute-id
     virtual-storage-desc:
       - storage-id
     sw-image-desc: image-test2
     configurable-properties:
       - key: "property1"
         value: "latest"
       - key: "property2"
         value: "edge"
"""

vdu_invalid_image = """
---
vdu:
   - id: test-vdu-id
     name: test-vdu-name
     int-cpd:
       - id: internal
         int-virtual-link-desc: network1
       - id: mgmt
     virtual-compute-desc: compute-id
     virtual-storage-desc:
       - storage-id
     sw-image-desc: invalid_image
     configurable-properties:
       - key: "property1"
         value: "latest"
       - key: "property2"
         value: "edge"
"""

sw_image_desc_nominal = """
---
sw-image-desc:
  - id: image-test1
    name: charm-name1
    image: ch:mysql
    version: lastest/edge
  - id: image-test2
    name: charm-name2
    image: ch:my-charm
    version: 8.0/stable
"""

sw_image_desc_no_channel = """
---
sw-image-desc:
  - id: image-test1
    name: charm-name1
    image: ch:mysql
  - id: image-test2
    name: charm-name2
    image: ch:my-charm
"""


class TestCharmInfoUtils(TestCase):
    def setUp(self):
        self.charm_info_utils = CharmInfoUtils()

    def get_loaded_descriptor(self, descriptor):
        return yaml.load(descriptor, Loader=yaml.Loader)

    def test_get_charm_info_nominal_case(self):
        vdu_descriptor = self.get_loaded_descriptor(vdu_nominal).get("vdu")
        sw_image_descs = self.get_loaded_descriptor(sw_image_desc_nominal).get(
            "sw-image-desc"
        )
        result = self.charm_info_utils.get_charm_info(vdu_descriptor[0], sw_image_descs)
        expected = CharmInfo("test-vdu-id", "8.0/stable", "ch:my-charm")
        self.assertEqual(result, expected)

    def test_get_charm_info_no_channel(self):
        vdu_descriptor = self.get_loaded_descriptor(vdu_nominal).get("vdu")
        sw_image_descs = self.get_loaded_descriptor(sw_image_desc_no_channel).get(
            "sw-image-desc"
        )
        result = self.charm_info_utils.get_charm_info(vdu_descriptor[0], sw_image_descs)
        expected = CharmInfo("test-vdu-id", None, "ch:my-charm")
        self.assertEqual(result, expected)

    def test_get_charm_info_invalid_image(self):
        vdu_descriptor = self.get_loaded_descriptor(vdu_invalid_image).get("vdu")
        sw_image_descs = self.get_loaded_descriptor(sw_image_desc_nominal).get(
            "sw-image-desc"
        )
        result = self.charm_info_utils.get_charm_info(vdu_descriptor[0], sw_image_descs)
        expected = CharmInfo("test-vdu-id", None, None)
        self.assertEqual(result, expected)

    def test_get_charm_info_empty_sw_image_descs(self):
        vdu_descriptor = self.get_loaded_descriptor(vdu_nominal).get("vdu")
        sw_image_descs = []
        result = self.charm_info_utils.get_charm_info(vdu_descriptor[0], sw_image_descs)
        expected = CharmInfo("test-vdu-id", None, None)
        self.assertEqual(result, expected)
