#######################################################################################
# Copyright ETSI Contributors and Others.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from datetime import timedelta
import traceback

from osm_common.temporal.activities.paas import (
    CheckCharmStatus,
    CheckCharmIsRemoved,
    DeployCharm,
    RemoveCharm,
    ResolveCharmErrors,
)
from osm_common.temporal.workflows.vdu import (
    VduInstantiateWorkflow,
    VduTerminateWorkflow,
)
from osm_common.temporal_task_queues.task_queues_mappings import LCM_TASK_QUEUE

from temporalio import workflow
from temporalio.common import RetryPolicy
from temporalio.exceptions import ActivityError

_SANDBOXED = False
retry_policy = RetryPolicy(maximum_attempts=3)
default_schedule_to_close_timeout = timedelta(minutes=10)


@workflow.defn(name=VduInstantiateWorkflow.__name__, sandboxed=_SANDBOXED)
class VduInstantiateWorkflowImpl(VduInstantiateWorkflow):
    @workflow.run
    async def run(self, workflow_input: VduInstantiateWorkflow.Input) -> None:
        deploy_charm_input = DeployCharm.Input(
            vim_uuid=workflow_input.vim_uuid,
            model_name=workflow_input.model_name,
            charm_info=workflow_input.charm_info,
            constraints=workflow_input.constraints,
            cloud=workflow_input.cloud,
            config=workflow_input.config,
        )
        try:
            self.logger.info(f"Deploying VDU `{workflow_input.charm_info.app_name}`")
            await workflow.execute_activity(
                activity=DeployCharm.__name__,
                arg=deploy_charm_input,
                activity_id=f"{DeployCharm.__name__}-{workflow_input.vim_uuid}",
                task_queue=LCM_TASK_QUEUE,
                schedule_to_close_timeout=default_schedule_to_close_timeout,
                retry_policy=retry_policy,
            )

            self.logger.info(
                f"Waiting for VDU `{workflow_input.charm_info.app_name}` to become ready"
            )
            await workflow.execute_activity(
                activity=CheckCharmStatus.__name__,
                arg=CheckCharmStatus.Input(
                    vim_uuid=workflow_input.vim_uuid,
                    model_name=workflow_input.model_name,
                    application_name=workflow_input.charm_info.app_name,
                ),
                activity_id=f"{CheckCharmStatus.__name__}-{workflow_input.vim_uuid}",
                task_queue=LCM_TASK_QUEUE,
                start_to_close_timeout=timedelta(minutes=5),
                heartbeat_timeout=timedelta(seconds=30),
                retry_policy=retry_policy,
            )

            self.logger.info(f"VDU `{workflow_input.charm_info.app_name}` is ready")

        except ActivityError as e:
            err_details = str(e.cause.with_traceback(e.__traceback__))
            self.logger.error(
                f"{VduInstantiateWorkflow.__name__} failed with {err_details}"
            )
            raise e

        except Exception as e:
            err_details = str(traceback.format_exc())
            self.logger.error(
                f"{VduInstantiateWorkflow.__name__} failed with {err_details}"
            )
            raise e


@workflow.defn(name=VduTerminateWorkflow.__name__, sandboxed=_SANDBOXED)
class VduTerminateWorkflowImpl(VduTerminateWorkflow):
    @workflow.run
    async def run(self, workflow_input: VduTerminateWorkflow.Input) -> None:
        try:
            vim_uuid = workflow_input.vim_uuid
            model_name = workflow_input.model_name
            application_name = workflow_input.application_name

            self.logger.info(f"Terminating VDU `{application_name}`.")
            try:
                await VduTerminateWorkflowImpl.remove_application_and_storage(
                    vim_uuid, model_name, application_name, False
                )
                self.logger.info(
                    f"Waiting for VDU `{workflow_input.application_name}` to be removed."
                )
            except ActivityError as remove_application_error:
                raise remove_application_error

            try:
                await VduTerminateWorkflowImpl.check_charm_is_removed(
                    vim_uuid, model_name, application_name
                )
                self.logger.info(f"VDU `{application_name}` is removed.")
                return

            except ActivityError:
                try:
                    self.logger.info(f"Resolving VDU `{application_name}` errors.")
                    await VduTerminateWorkflowImpl.resolve_charm_errors(
                        vim_uuid, model_name, application_name
                    )

                except ActivityError:
                    try:
                        self.logger.info(
                            f"Removing VDU `{application_name}` forcefully."
                        )
                        await VduTerminateWorkflowImpl.remove_application_and_storage(
                            vim_uuid, model_name, application_name, True
                        )

                    except ActivityError as remove_application_error:
                        raise remove_application_error

                    try:
                        await VduTerminateWorkflowImpl.check_charm_is_removed(
                            vim_uuid, model_name, application_name
                        )
                        self.logger.info(f"VDU `{application_name}` is removed.")
                        return

                    except ActivityError as check_charm_removal_error:
                        self.logger.error(
                            f"VDU {application_name} could not be removed."
                        )
                        raise check_charm_removal_error

                try:
                    self.logger.info(
                        f"Removing VDU `{application_name}` after error resolution."
                    )
                    await VduTerminateWorkflowImpl.remove_application_and_storage(
                        vim_uuid, model_name, application_name, False
                    )
                except ActivityError as remove_application_error:
                    raise remove_application_error

                try:
                    await VduTerminateWorkflowImpl.check_charm_is_removed(
                        vim_uuid, model_name, application_name
                    )
                    self.logger.info(f"VDU `{application_name}` is removed.")
                    return

                except ActivityError:
                    self.logger.info(f"Removing VDU `{application_name}` forcefully.")
                    await VduTerminateWorkflowImpl.remove_application_and_storage(
                        vim_uuid, model_name, application_name, True
                    )

                    try:
                        await VduTerminateWorkflowImpl.check_charm_is_removed(
                            vim_uuid, model_name, application_name
                        )
                        self.logger.info(f"VDU `{application_name}` is removed.")
                        return

                    except ActivityError as check_charm_removal_error:
                        self.logger.error(
                            f"VDU {application_name} could not be removed."
                        )
                        raise check_charm_removal_error

        except ActivityError as e:
            err_details = str(e.cause.with_traceback(e.__traceback__))
            self.logger.error(
                f"{VduTerminateWorkflow.__name__} failed with {err_details}"
            )
            raise e

        except Exception as e:
            err_details = str(traceback.format_exc())
            self.logger.error(
                f"{VduTerminateWorkflow.__name__} failed with {err_details}"
            )
            raise e

    @staticmethod
    async def remove_application_and_storage(
        vim_uuid, model_name, application_name, force_remove
    ):
        await workflow.execute_activity(
            activity=RemoveCharm.__name__,
            arg=RemoveCharm.Input(
                vim_uuid=vim_uuid,
                model_name=model_name,
                application_name=application_name,
                force_remove=force_remove,
            ),
            activity_id=f"{RemoveCharm.__name__}-{vim_uuid}",
            task_queue=LCM_TASK_QUEUE,
            schedule_to_close_timeout=default_schedule_to_close_timeout,
            retry_policy=retry_policy,
        )

    @staticmethod
    async def check_charm_is_removed(vim_uuid, model_name, application_name):
        await workflow.execute_activity(
            activity=CheckCharmIsRemoved.__name__,
            arg=CheckCharmIsRemoved.Input(
                vim_uuid=vim_uuid,
                model_name=model_name,
                application_name=application_name,
            ),
            activity_id=f"{CheckCharmIsRemoved.__name__}-{vim_uuid}",
            task_queue=LCM_TASK_QUEUE,
            start_to_close_timeout=timedelta(minutes=5),
            heartbeat_timeout=timedelta(seconds=30),
            retry_policy=retry_policy,
        )

    @staticmethod
    async def resolve_charm_errors(vim_uuid, model_name, application_name):
        await workflow.execute_activity(
            activity=ResolveCharmErrors.__name__,
            arg=ResolveCharmErrors.Input(
                vim_uuid=vim_uuid,
                model_name=model_name,
                application_name=application_name,
            ),
            activity_id=f"{ResolveCharmErrors.__name__}-{vim_uuid}",
            task_queue=LCM_TASK_QUEUE,
            start_to_close_timeout=timedelta(minutes=3),
            heartbeat_timeout=timedelta(seconds=30),
            retry_policy=retry_policy,
        )
