# Copyright 2023 Canonical Ltd.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import logging


class VnfModelHandler:

    logger = logging.getLogger("lcm_vnf")

    @staticmethod
    def get_vnf_model_id(nsr_id: str, vim_id: str, vnf_descriptor: dict) -> str:
        """Return VNF model ID if model name specified in descriptor. NS default model ID otherwhise."""
        vnf_model_name = VnfModelHandler._get_vnf_model_name(vnf_descriptor)
        if vnf_model_name:
            suffix = nsr_id[-4:]
            return vnf_model_name + "-" + suffix
        return VnfModelHandler._get_ns_default_model_id(nsr_id, vim_id)

    @staticmethod
    def _get_ns_default_model_id(nsr_id: str, vim_id: str) -> str:
        """The NS default model ID is the combination if the NSr ID and the VIM ID."""
        return nsr_id[-12:] + "-" + vim_id[-12:]

    @classmethod
    def _get_vnf_model_name(cls, vnf_descriptor: dict) -> str:
        """Return the model name if specified in VNF descriptor. Return None otherwhise."""
        try:
            additional_properties = (
                vnf_descriptor.get("vnfd")
                .get("configurable-properties")
                .get("additional-configurable-property")
            )
            for vnf_property in additional_properties:
                if vnf_property.get("key") == "model":
                    return vnf_property.get("value")
            return None
        except Exception:
            cls.logger.info(
                "Model name not specified in VNF descriptor. NS model will be used."
            )
            return None
