# Copyright 2023 Canonical Ltd.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from datetime import timedelta
from osm_lcm.temporal.vim_activities import (
    ACTIVITY_GET_JUJU_CONTROLLER_DEFINITION,
    ACTIVITY_TEST_VIM_CONNECTIVITY,
    ACTIVITY_UPDATE_VIM_STATUS,
)
from temporal.libjuju import ConnectionInfo
from temporalio import workflow
from temporalio.common import RetryPolicy
from temporalio.converter import value_to_type
from temporalio.exceptions import ActivityError, ApplicationError

from osm_lcm.temporal.names import lcm_task_queue


do_not_retry_policy = RetryPolicy(maximum_attempts=3)
default_schedule_to_close_timeout = timedelta(seconds=1000)


@workflow.defn(name="vim-create", sandboxed=False)
class VimCreateWorkflow:
    @workflow.run
    async def run(self, content: dict) -> None:
        try:
            vim_id = content["_id"]
            connection_info = value_to_type(
                ConnectionInfo,
                await workflow.execute_activity(
                    activity=ACTIVITY_GET_JUJU_CONTROLLER_DEFINITION,
                    arg=content,
                    activity_id=f"{ACTIVITY_GET_JUJU_CONTROLLER_DEFINITION}-{vim_id}",
                    task_queue=lcm_task_queue,
                    schedule_to_close_timeout=default_schedule_to_close_timeout,
                    retry_policy=do_not_retry_policy,
                ),
            )

            await workflow.execute_activity(
                activity=ACTIVITY_TEST_VIM_CONNECTIVITY,
                arg=connection_info,
                activity_id=f"{ACTIVITY_TEST_VIM_CONNECTIVITY}-{vim_id}",
                task_queue=lcm_task_queue,
                schedule_to_close_timeout=default_schedule_to_close_timeout,
                retry_policy=do_not_retry_policy,
                heartbeat_timeout=timedelta(seconds=200),
            )

            # TODO: Revisit if we should move `_get_vim_create_update`to a separate
            # activity, or if this logic is simple enough to do in the workflow.
            # Note: If this fails, the workflow *task* will fail (but not the
            # workflow) and the workflow will be retried indefinitely.
            #
            # See: https://docs.temporal.io/kb/failures#throw-from-workflows
            db_vim_update = self._get_vim_create_update(
                content["op_id"], "SUCCESS", "VIM created successfully."
            )

        except ActivityError as e:
            db_vim_update = self._get_vim_create_update(
                content["op_id"], "FAILED", e.cause.message
            )
            raise e

        except Exception as e:
            db_vim_update = self._get_vim_create_update(
                content["op_id"], "FAILED", str(e)
            )
            raise e
        finally:
            db_update_info = {"id": vim_id, "db_update": db_vim_update}
            await workflow.execute_activity(
                activity=ACTIVITY_UPDATE_VIM_STATUS,
                arg=db_update_info,
                activity_id=f"{ACTIVITY_UPDATE_VIM_STATUS}-{vim_id}",
                task_queue=lcm_task_queue,
                schedule_to_close_timeout=default_schedule_to_close_timeout,
                retry_policy=do_not_retry_policy,
            )

    def _get_op_index(self, op_id: str) -> str:
        _, _, op_index = op_id.rpartition(":")
        return op_index

    def _get_vim_create_update(self, op_id: str, code: str, message: str) -> dict:
        op_index = self._get_op_index(op_id)
        return {
            "_admin.operationalState": code,
            "_admin.detailed-status": message,
            "_admin.current_operation": None,
            "_admin.operations.{}.operationState".format(op_index): code,
            "_admin.operations.{}.detailed-status".format(op_index): message,
        }
