# Copyright 2023 Canonical Ltd.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
from functools import wraps
from temporalio.exceptions import ApplicationError
from time import time
from temporalio import activity

from osm_lcm.temporal.libjuju import ConnectionInfo, Libjuju


activity.logger = logging.getLogger("lcm.temporal.activities")


ACTIVITY_TEST_VIM_CONNECTIVITY = "test-vim-connectivity"
ACTIVITY_UPDATE_VIM_STATUS = "update-vim-status"
ACTIVITY_GET_JUJU_CONTROLLER_DEFINITION = "get-juju-controller-connection-info"


class RetryableException(ApplicationError):
    def __init__(self, message):
        super().__init__(message, non_retryable=False)


def wrap_exceptions(error_message="Unhandled exception"):
    def Inner(f):
        @wraps(f)
        async def wrapper(*args, **kwargs):
            try:
                return await f(*args, **kwargs)
            except Exception as e:
                if isinstance(e, RetryableException):
                    raise e
                raise ApplicationError(
                    f"{error_message}: {str(e)}", non_retryable=True
                ) from e

        return wrapper

    return Inner


class JujuPaasConnector:
    @staticmethod
    @activity.defn(name=ACTIVITY_TEST_VIM_CONNECTIVITY)
    @wrap_exceptions(error_message="Cannot connect to juju controller")
    async def test_vim_connectivity(
        connection_info: ConnectionInfo,
    ) -> None:
        controller = None
        libjuju = Libjuju(connection_info)
        try:
            # await sleep(1) # 1s
            # activity.heartbeat("Executed 1s operations")
            # await sleep(5) # 5s
            # activity.heartbeat("Executed 5s operations")
            # await sleep(50) # 50s
            # activity.heartbeat("Executed 50s operations")
            # await sleep(100) # 100s
            # activity.heartbeat("Executed 100s operations")
            # await sleep(150) # 150s
            activity.heartbeat("Getting Juju controller")
            controller = await libjuju.get_controller()
            activity.heartbeat("Got Juju controller")
        finally:
            await libjuju.disconnect_controller(controller)


class VimDbActivity:
    def __init__(self, db):
        self.db = db

    @activity.defn(name=ACTIVITY_UPDATE_VIM_STATUS)
    @wrap_exceptions(error_message="Error writing status")
    async def update_vim_status(self, db_update_info: dict) -> None:
        vim_id = db_update_info.get("id")
        db_update = db_update_info.get("db_update")
        db_update["_admin.modified"] = time()
        self.db.set_one("vim_accounts", {"_id": vim_id}, db_update)
        message = "VIM {} is updated.".format(vim_id)
        activity.logger.info(message)


    def _decrypt_password(self, vim_content: dict) -> str:
        try:
            return self.db.decrypt(
                vim_content["vim_password"],
                schema_version=vim_content["schema_version"],
                salt=vim_content["_id"],
            )
        except Exception as e:
            activity.logger.error(
                "Error decrypting password, vim={}: {}".format(
                    vim_content.get("_id"), e
                )
            )
            raise e

    @activity.defn(name=ACTIVITY_GET_JUJU_CONTROLLER_DEFINITION)
    @wrap_exceptions(error_message="Invalid VIM connection info")
    async def get_juju_controller_connection_info(
        self, vim_content: dict
    ) -> ConnectionInfo:
        endpoint = vim_content["vim_url"]
        username = vim_content["vim_user"]
        password = self._decrypt_password(vim_content)
        vim_config = vim_content["config"]
        cacert = vim_config["ca_cert_content"]
        cloud_name = vim_config["cloud"]
        cloud_credentials = vim_config["cloud_credentials"]
        return ConnectionInfo(
            endpoint, username, password, cacert, cloud_name, cloud_credentials
        )
