# Copyright 2023 Canonical Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
#     Unless required by applicable law or agreed to in writing, software
#     distributed under the License is distributed on an "AS IS" BASIS,
#     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#     See the License for the specific language governing permissions and
#     limitations under the License.

from unittest import TestCase
from osm_lcm.temporal.vnf_model_handler import VnfModelHandler
import yaml

nsr_id = "ea958ba5-4e58-4405-bf42-6e3be15d4c3a"
vim_id = "70b47595-fafa-4f63-904b-fc3ada60eebb"
expected_default_ns_model = "6e3be15d4c3a-fc3ada60eebb"

vnf_descriptor_model_key = """
---
vnfd:
  id: sol006-vnf
  configurable-properties:
    additional-configurable-property:
      - key: track
        value: latest
      - key: model
        value: vnf-model
"""

vnf_descriptor_no_model_key = """
---
vnfd:
  id: sol006-vnf
  configurable-properties:
    additional-configurable-property:
      - key: track
        value: latest
"""

vnf_descriptor_no_additional_configurable_properties = """
---
vnfd:
  id: sol006-vnf
  configurable-properties:
    some-property:
      - key: model
        value: vnf-model
"""

vnf_descriptor_no_configurable_properties = """
---
vnfd:
  id: sol006-vnf
  provider: Canonical
"""


class TestVnfModelHandler(TestCase):
    def setUp(self):
        self.vnf_handler = VnfModelHandler()

    def get_loaded_descriptor(self, vnf_descriptor):
        return yaml.load(vnf_descriptor, Loader=yaml.Loader)

    def test_get_vnf_model_id_model_name_in_descriptor(self):
        vnf_descriptor = self.get_loaded_descriptor(vnf_descriptor_model_key)
        result = self.vnf_handler.get_vnf_model_id(nsr_id, vim_id, vnf_descriptor)
        expected = "vnf-model-4c3a"
        self.assertEqual(result, expected)

    def test_get_vnf_model_id_no_model_key_in_descriptor(self):
        vnf_descriptor = self.get_loaded_descriptor(vnf_descriptor_no_model_key)
        result = self.vnf_handler.get_vnf_model_id(nsr_id, vim_id, vnf_descriptor)
        self.assertEqual(result, expected_default_ns_model)

    def test_get_vnf_model_id_no_additional_configurable_property(self):
        vnf_descriptor = self.get_loaded_descriptor(
            vnf_descriptor_no_additional_configurable_properties
        )
        result = self.vnf_handler.get_vnf_model_id(nsr_id, vim_id, vnf_descriptor)
        self.assertEqual(result, expected_default_ns_model)

    def test_get_vnf_model_id_no_configurable_properties(self):
        vnf_descriptor = self.get_loaded_descriptor(
            vnf_descriptor_no_configurable_properties
        )
        result = self.vnf_handler.get_vnf_model_id(nsr_id, vim_id, vnf_descriptor)
        self.assertEqual(result, expected_default_ns_model)
