/* Copyright 2017 Sandvine
 *
 * All Rights Reserved.
 *
 *   Licensed under the Apache License, Version 2.0 (the "License"); you may
 *   not use this file except in compliance with the License. You may obtain
 *   a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *   WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *   License for the specific language governing permissions and limitations
 *   under the License.
 */

properties([
    parameters([
        string(defaultValue: env.GERRIT_BRANCH, description: '', name: 'GERRIT_BRANCH'),
        string(defaultValue: 'system', description: '', name: 'NODE'),
        string(defaultValue: '', description: '', name: 'BUILD_FROM_SOURCE'),
        string(defaultValue: 'unstable', description: '', name: 'REPO_DISTRO'),
        string(defaultValue: '', description: '', name: 'COMMIT_ID'),
        string(defaultValue: '-stage_2', description: '', name: 'UPSTREAM_SUFFIX'),
        string(defaultValue: 'pubkey.asc', description: '', name: 'REPO_KEY_NAME'),
        string(defaultValue: 'release', description: '', name: 'RELEASE'),
        string(defaultValue: '', description: '', name: 'UPSTREAM_JOB_NAME'),
        string(defaultValue: '', description: '', name: 'UPSTREAM_JOB_NUMBER'),
        string(defaultValue: '', description: '', name: 'UPSTREAM_JOB_NUMBER'),
        string(defaultValue: 'OSMETSI', description: '', name: 'GPG_KEY_NAME'),
        string(defaultValue: 'artifactory-osm', description: '', name: 'ARTIFACTORY_SERVER'),
        string(defaultValue: 'osm-stage_4', description: '', name: 'DOWNSTREAM_STAGE_NAME'),
        string(defaultValue: 'testing-daily', description: '', name: 'DOCKER_TAG'),
        booleanParam(defaultValue: true, description: '', name: 'SAVE_CONTAINER_ON_FAIL'),
        booleanParam(defaultValue: false, description: '', name: 'SAVE_CONTAINER_ON_PASS'),
        booleanParam(defaultValue: true, description: '', name: 'SAVE_ARTIFACTS_ON_SMOKE_SUCCESS'),
        booleanParam(defaultValue: true, description: '', name: 'DO_STAGE_4'),
        booleanParam(defaultValue: true, description: '',  name: 'DO_BUILD'),
        booleanParam(defaultValue: true, description: '', name: 'DO_INSTALL'),
        booleanParam(defaultValue: true, description: '', name: 'DO_SMOKE'),
        booleanParam(defaultValue: true, description: '', name: 'DO_DOCKERPUSH'),
        booleanParam(defaultValue: false, description: '', name: 'SAVE_ARTIFACTS_OVERRIDE'),
        string(defaultValue: '/home/jenkins/hive/openstack-etsi.rc', description: '', name: 'HIVE_VIM_1'),
        booleanParam(defaultValue: false, description: '', name: 'DO_ROBOT'),
        string(defaultValue: 'sanity', description: 'sanity/regression are the options', name: 'TEST_NAME'),
        string(defaultValue: '/home/jenkins/hive/robot-systest.cfg', description: '', name: 'ROBOT_VIM'),
        string(defaultValue: '/home/jenkins/hive/kubeconfig.yaml', description: '', name: 'KUBECONFIG'),
        string(defaultValue: '/home/jenkins/hive/clouds.yaml', description: '', name: 'CLOUDS'),
        string(defaultValue: 'Charmed', description: '', name: 'INSTALLER'),
    ])
])

def uninstall_osm(stackName) {
    sh """
         export OSM_USE_LOCAL_DEVOPS=true
         export PATH=$PATH:/snap/bin
         installers/full_install_osm.sh -y -w /tmp/osm -t ${stackName} -s ${stackName} --test --nolxd --nodocker --nojuju --nohostports --nohostclient --uninstall
       """
}

def run_systest(stackName,tagName,testName,envfile=null) {
    tempdir = sh(returnStdout: true, script: "mktemp -d").trim()
    if ( !envfile )
    {
        sh(script: "touch ${tempdir}/env")
        envfile="${tempdir}/env"
    }
    sh "docker run --network net${stackName} --env-file ${envfile} -v ${tempdir}:/usr/share/osm-devops/systest/reports opensourcemano/osmclient:${tagName} make -C /usr/share/osm-devops/systest ${testName}"
    sh "cp ${tempdir}/* ."
    junit  '*.xml'
}

def run_robot_systest(tagName,testName,osmHostname,prometheusHostname,prometheus_port=null,envfile=null,kubeconfig=null,clouds=null,hostfile=null) {
    tempdir = sh(returnStdout: true, script: "mktemp -d").trim()
    if ( !envfile )
    {
        sh(script: "touch ${tempdir}/env")
        envfile="${tempdir}/env"
    }
    PROMETHEUS_PORT_VAR = ""
    if ( prometheusPort != null) {
        PROMETHEUS_PORT_VAR = "--env PROMETHEUS_PORT="+prometheusPort
    }
    hostfilemount=""
    if ( hostfile ) {
        hostfilemount="-v "+hostfile+":/etc/hosts"
    }

    try {
        sh "docker run --env OSM_HOSTNAME=${osmHostname} --env PROMETHEUS_HOSTNAME=${prometheusHostname} ${PROMETHEUS_PORT_VAR} --env-file ${envfile} -v ${clouds}:/etc/openstack/clouds.yaml -v ${kubeconfig}:/root/.kube/config -v ${tempdir}:/robot-systest/reports ${hostfilemount} opensourcemano/tests:${tagName} -c -t ${testName}"
    } finally {
        sh "cp ${tempdir}/* ."
        outputDirectory = sh(returnStdout: true, script: "pwd").trim()
        println ("Present Directory is : ${outputDirectory}")
        step([
            $class : 'RobotPublisher',
            outputPath : "${outputDirectory}",
            outputFileName : "*.xml",
            disableArchiveOutput : false,
            reportFileName : "report.html",
            logFileName : "log.html",
            passThreshold : 0,
            unstableThreshold: 0,
            otherFiles : "*.png",
        ])
    }
}

def archive_logs(remote) {

    sshCommand remote: remote, command: '''mkdir -p logs'''
    if (useCharmedInstaller) {
        sshCommand remote: remote, command: '''
            for container in `kubectl get pods -n osm | grep -v operator | grep -v NAME| awk '{print $1}'`; do
                logfile=`echo $container | cut -d- -f1`
                echo "Extracting log for $logfile"
                kubectl logs -n osm $container 2>&1 > logs/$logfile.log
            done
        '''
    } else {
        // collect logs from k8s based installer...
    }

    sh "rm -rf logs"
    sshCommand remote: remote, command: '''ls -al logs'''
    sshGet remote: remote, from: 'logs', into: '.', override: true
    sh "cp logs/* ."
    archiveArtifacts artifacts: '*.log'
}

def get_value(key, output) {
    for (String line : output.split( '\n' )) {
        data = line.split( '\\|' )
        if (data.length > 1) {
            if ( data[1].trim() == key ) {
                return data[2].trim()
            }
        }
    }
}

node("${params.NODE}") {

    INTERNAL_DOCKER_REGISTRY = 'osm.etsi.org:5050/devops/cicd/'
    SSH_KEY = '~/hive/cicd_rsa'
    sh 'env'

    tag_or_branch = params.GERRIT_BRANCH.replaceAll(/\./,"")

    stage("Checkout") {
        checkout scm
    }

    ci_helper = load "jenkins/ci-pipelines/ci_helper.groovy"

    def upstream_main_job = params.UPSTREAM_SUFFIX

    // upstream jobs always use merged artifacts
    upstream_main_job += '-merge'
    container_name_prefix = "osm-${tag_or_branch}"
    container_name = "${container_name_prefix}"

    keep_artifacts = false
    if ( JOB_NAME.contains('merge') ) {
        container_name += "-merge"

        // On a merge job, we keep artifacts on smoke success
        keep_artifacts = params.SAVE_ARTIFACTS_ON_SMOKE_SUCCESS
    }
    container_name += "-${BUILD_NUMBER}"

    // Copy the artifacts from the upstream jobs
    stage("Copy Artifacts") {
        // cleanup any previous repo
        sh 'rm -rf repo'
        dir("repo") {
            // grab all stable upstream builds based on the

            dir("${RELEASE}") {
                def list = ["RO", "openvim", "osmclient", "IM", "devops", "MON", "N2VC", "NBI", "common", "LCM", "POL", "LW-UI", "NG-UI", "PLA", "tests"]
                for (component in list) {
                    step ([$class: 'CopyArtifact',
                           projectName: "${component}${upstream_main_job}/${GERRIT_BRANCH}"])

                    // grab the build name/number
                    build_num = ci_helper.get_env_value('build.env','BUILD_NUMBER')

                    // grab the archives from the stage_2 builds (ie. this will be the artifacts stored based on a merge)
                    ci_helper.get_archive(params.ARTIFACTORY_SERVER,component,GERRIT_BRANCH, "${component}${upstream_main_job} :: ${GERRIT_BRANCH}", build_num)

                    // cleanup any prevously defined dists
                    sh "rm -rf dists"
                }

                // check if an upstream artifact based on specific build number has been requested
                // This is the case of a merge build and the upstream merge build is not yet complete (it is not deemed
                // a successful build yet). The upstream job is calling this downstream job (with the its build artifiact)
                if ( params.UPSTREAM_JOB_NAME ) {
                    step ([$class: 'CopyArtifact',
                           projectName: "${params.UPSTREAM_JOB_NAME}",
                           selector: [$class: 'SpecificBuildSelector', buildNumber: "${params.UPSTREAM_JOB_NUMBER}"]
                          ])

                    build_num = ci_helper.get_env_value('build.env','BUILD_NUMBER')
                    component = ci_helper.get_mdg_from_project(ci_helper.get_env_value('build.env','GERRIT_PROJECT'))

                    // the upstream job name contains suffix with the project. Need this stripped off
                    def project_without_branch = params.UPSTREAM_JOB_NAME.split('/')[0]

                    // Remove the previous artifact for this component. Use the new upstream artifact
                    sh "rm -rf pool/${component}"

                    ci_helper.get_archive(params.ARTIFACTORY_SERVER,component,GERRIT_BRANCH, "${project_without_branch} :: ${GERRIT_BRANCH}", build_num)

                    sh "rm -rf dists"
                }

                // sign all the components
                for (component in list) {
                    sh "dpkg-sig --sign builder -k ${GPG_KEY_NAME} pool/${component}/*"
                }

                // now create the distro
                for (component in list) {
                    sh "mkdir -p dists/${params.REPO_DISTRO}/${component}/binary-amd64/"
                    sh "apt-ftparchive packages pool/${component} > dists/${params.REPO_DISTRO}/${component}/binary-amd64/Packages"
                    sh "gzip -9fk dists/${params.REPO_DISTRO}/${component}/binary-amd64/Packages"
                }

                // create and sign the release file
                sh "apt-ftparchive release dists/${params.REPO_DISTRO} > dists/${params.REPO_DISTRO}/Release"
                sh "gpg --yes -abs -u ${GPG_KEY_NAME} -o dists/${params.REPO_DISTRO}/Release.gpg dists/${params.REPO_DISTRO}/Release"

                // copy the public key into the release folder
                // this pulls the key from the home dir of the current user (jenkins)
                sh "cp ~/${REPO_KEY_NAME} 'OSM ETSI Release Key.gpg'"
                sh "cp ~/${REPO_KEY_NAME} ."

                // merge the change logs
                sh """
                   rm -f changelog/changelog-osm.html
                   [ ! -d changelog ] || for mdgchange in \$(ls changelog); do cat changelog/\$mdgchange >> changelog/changelog-osm.html; done
                   """
                RELEASE_DIR = sh(returnStdout:true,  script: 'pwd').trim()
            }
            // start an apache server to serve up the images
            http_server_name = "${container_name}-apache"

            pwd = sh(returnStdout:true,  script: 'pwd').trim()
            repo_port = sh(script: 'echo $(python -c \'import socket; s=socket.socket(); s.bind(("", 0)); print(s.getsockname()[1]); s.close()\');', returnStdout: true).trim()
            repo_base_url = ci_helper.start_http_server(pwd,http_server_name,repo_port)
            NODE_IP_ADDRESS=sh(returnStdout: true, script:
                "echo ${SSH_CONNECTION} | awk '{print \$3}'").trim()
        }

        // now pull the devops package and install in temporary location
        tempdir = sh(returnStdout: true, script: "mktemp -d").trim()
        osm_devops_dpkg = sh(returnStdout: true, script: "find . -name osm-devops*.deb").trim()
        sh "dpkg -x ${osm_devops_dpkg} ${tempdir}"
        OSM_DEVOPS="${tempdir}/usr/share/osm-devops"
        println("Repo base URL=${repo_base_url}")
    }

    dir(OSM_DEVOPS) {
        def remote = [:]
        error = null

        if ( params.DO_BUILD ) {
            stage("Build") {
                sh "make -C docker clean"
                sh "make -C docker -j `nproc` Q= CMD_DOCKER_ARGS= TAG=${container_name} RELEASE=${params.RELEASE} REPOSITORY_BASE=${repo_base_url} REPOSITORY_KEY=${params.REPO_KEY_NAME} REPOSITORY=${params.REPO_DISTRO}"
            }

            stage("Push to internal registry") {
                withCredentials([[$class: 'UsernamePasswordMultiBinding', credentialsId: 'gitlab-registry',
                                usernameVariable: 'USERNAME', passwordVariable: 'PASSWORD']]) {
                    sh "docker login ${INTERNAL_DOCKER_REGISTRY} -u ${USERNAME} -p ${PASSWORD}"
                }
                sh "make -C docker push INPUT_TAG=${container_name} TAG=${container_name} DOCKER_REGISTRY=${INTERNAL_DOCKER_REGISTRY}"
            }

        }

        try {
            useCharmedInstaller = params.INSTALLER.equalsIgnoreCase("charmed")

            if ( params.DO_INSTALL ) {

                stage("Spawn Remote VM") {
                    println("Launching new VM")
                    output=sh(returnStdout: true, script: """#!/bin/sh -e
                        for line in `grep OS ~/hive/robot-systest.cfg | grep -v OS_CLOUD` ; do export \$line ; done
                        openstack server create --flavor m1.xlarge \
                                                --image ubuntu18.04 \
                                                --key-name CICD \
                                                --nic net-id=osm-ext \
                                                ${container_name}
                    """).trim()

                    server_id = get_value('id', output)

                    if (server_id == null) {
                        println("VM launch output: ")
                        println(output)
                        throw new Exception("VM Launch failed")
                    }
                    println("Target VM is ${server_id}, waiting for IP address to be assigned")

                    IP_ADDRESS = ""

                    while (IP_ADDRESS == "") {
                        output=sh(returnStdout: true, script: """#!/bin/sh -e
                            for line in `grep OS ~/hive/robot-systest.cfg | grep -v OS_CLOUD` ; do export \$line ; done
                            openstack server show ${server_id}
                        """).trim()
                        IP_ADDRESS = get_value('addresses', output)
                    }
                    IP_ADDRESS = IP_ADDRESS.split('=')[1]
                    println("Waiting for VM at ${IP_ADDRESS} to be reachable")

                    alive = false
                    while (! alive) {
                        output=sh(returnStdout: true, script: "sleep 1 ; nc -zv ${IP_ADDRESS} 22 2>&1 || true").trim()
                        println("output is [$output]")
                        alive = output.contains("succeeded")
                    }
                    println("VM is ready and accepting ssh connections")
                }

                stage("Install") {
                    commit_id = ''
                    repo_distro = ''
                    repo_key_name = ''
                    release = ''

                    if ( params.COMMIT_ID )
                    {
                        commit_id = "-b ${params.COMMIT_ID}"
                    }

                    if ( params.REPO_DISTRO )
                    {
                        repo_distro = "-r ${params.REPO_DISTRO}"
                    }

                    if ( params.REPO_KEY_NAME )
                    {
                        repo_key_name = "-k ${params.REPO_KEY_NAME}"
                    }

                    if ( params.RELEASE )
                    {
                        release = "-R ${params.RELEASE}"
                    }

                    if ( params.REPOSITORY_BASE )
                    {
                        repo_base_url = "-u ${params.REPOSITORY_BASE}"
                    }
                    else
                    {
                        repo_base_url = "-u http://${NODE_IP_ADDRESS}:${repo_port}"
                    }

                    remote.name = container_name
                    remote.host = IP_ADDRESS
                    remote.user = 'ubuntu'
                    remote.identityFile = SSH_KEY
                    remote.allowAnyHosts = true
                    remote.logLevel = 'INFO'

                    sshCommand remote: remote, command: """
                        wget https://osm-download.etsi.org/ftp/osm-8.0-eight/install_osm.sh
                        chmod +x ./install_osm.sh
                    """

                    if ( useCharmedInstaller ) {

                        // Use local proxy for docker hub
                        sshCommand remote: remote, command: '''
                            sudo snap install microk8s --classic --channel=1.19/stable
                            sudo sed -i "s|https://registry-1.docker.io|http://172.21.1.1:5000|" /var/snap/microk8s/current/args/containerd-template.toml
                            sudo systemctl restart snap.microk8s.daemon-containerd.service
                            sudo snap alias microk8s.kubectl kubectl
                            echo export PATH=/snap/bin:\${PATH} > ~/.bashrc
                        '''

                        withCredentials([[$class: 'UsernamePasswordMultiBinding', credentialsId: 'gitlab-registry',
                                        usernameVariable: 'USERNAME', passwordVariable: 'PASSWORD']]) {
                            sshCommand remote: remote, command: """
                                ./install_osm.sh -y \
                                    ${repo_base_url} \
                                    ${repo_key_name} \
                                    ${release} -r unstable \
                                    --charmed  \
                                    --registry ${USERNAME}:${PASSWORD}@${INTERNAL_DOCKER_REGISTRY} \
                                    --tag ${container_name}
                            """
                        }
                        prometheusHostname = "prometheus."+IP_ADDRESS+".xip.io"
                        prometheusPort = 80
                        osmHostname = "nbi."+IP_ADDRESS+".xip.io:443"
                    } else {
                        // Run -k8s installer here
                        // Update to use 172.21.1.1 as the dockerhub proxy
                        // Specify registry to use for installer
                        // set osmHostname, prometheusHostname, prometheusPort as needed
                    }
                }
            }

            stage_archive = false
            if ( params.DO_SMOKE ) {
                stage("OSM Health") {
                    if ( useCharmedInstaller ) {
                        stackName = "osm"
                    } else {
                        stackName = container_name
                    }
                    sshCommand remote: remote, command: """
                        /usr/share/osm-devops/installers/osm_health.sh -k -s ${stackName}
                    """
                }
            }

            if ( params.DO_STAGE_4 ) {
                // override stage_archive to only archive on stable
                stage_archive = false
                try {
                    stage("System Integration Test") {
                        if ( params.DO_ROBOT ) {
                            if( useCharmedInstaller ) {
                                tempdir = sh(returnStdout: true, script: "mktemp -d").trim()
                                sh(script: "touch ${tempdir}/hosts")
                                hostfile="${tempdir}/hosts"
                                sh """cat << EOF > ${hostfile}
127.0.0.1           localhost
${remote.host}      prometheus.${remote.host}.xip.io nbi.${remote.host}.xip.io
EOF"""
                            } else {
                                hostfile=null
                            }

                            run_robot_systest(
                                container_name,
                                params.TEST_NAME,
                                osmHostname,
                                prometheusHostname,
                                prometheusPort,
                                params.ROBOT_VIM,
                                params.KUBECONFIG,
                                params.CLOUDS,
                                hostfile)
                        }
                    }
                } finally {
                    stage("Archive Contailer Logs") {
                        // Archive logs to containers_logs.txt
                        archive_logs(remote)
                        if ( ! currentBuild.result.equals('UNSTABLE') && ! currentBuild.result.equals('FAILURE')) {
                            stage_archive = keep_artifacts
                        } else {
                            println ("Systest test failed, throwing error")
                            error = new Exception("Systest test failed")
                            currentBuild.result = 'FAILURE'
                            throw error
                        }
                    }
                }
            }

            // override to save the artifacts
            if ( params.SAVE_ARTIFACTS_OVERRIDE || stage_archive ) {
                stage("Archive") {
                    sh "echo ${container_name} > build_version.txt"
                    archiveArtifacts artifacts: "build_version.txt", fingerprint: true

                    // Archive the tested repo
                    dir("${RELEASE_DIR}") {
                        ci_helper.archive(params.ARTIFACTORY_SERVER,RELEASE,GERRIT_BRANCH,'tested')
                    }
                    if ( params.DO_DOCKERPUSH ) {
                        stage("Docker Push") {
                            sh "make -C docker push INPUT_TAG=${container_name} TAG=${params.DOCKER_TAG}"
                        }

                        stage("Snap promotion") {
                            def snaps = ["osmclient"]
                            for (snap in snaps) {
                                channel=""
                                if (BRANCH_NAME.startsWith("v")) {
                                    channel=BRANCH_NAME.substring(1)+"/"
                                } else if (BRANCH_NAME!="master") {
                                    channel+="/"+BRANCH_NAME.replaceAll('/','-')
                                }
                                track=channel+"edge\\*"
                                edge_rev=sh(returnStdout: true,
                                    script: "sudo docker run -v ~/.snapcraft:/snapcraft -v ${WORKSPACE}:/build " +
                                    "-w /build snapcore/snapcraft:stable /bin/bash -c " +
                                    "\"snapcraft login --with /snapcraft/config &>/dev/null && " +
                                    "snapcraft revisions $snap\" | " +
                                    "grep \" $track\" | tail -1 | awk '{print \$1}'").trim()
                                track=channel+"beta\\*"
                                beta_rev=sh(returnStdout: true,
                                    script: "sudo docker run -v ~/.snapcraft:/snapcraft -v ${WORKSPACE}:/build " +
                                    "-w /build snapcore/snapcraft:stable /bin/bash -c " +
                                    "\"snapcraft login --with /snapcraft/config &>/dev/null && " +
                                    "snapcraft revisions $snap\" | " +
                                    "grep \" $track\" | tail -1 | awk '{print \$1}'").trim()

                                if ( edge_rev != beta_rev ) {
                                    print "Promoting $edge_rev to beta in place of $beta_rev"
                                    beta_track=channel+"beta"
                                    sh("sudo docker run -v ~/.snapcraft:/snapcraft -v ${WORKSPACE}:/build " +
                                        "-w /build snapcore/snapcraft:stable /bin/bash -c " +
                                        "\"snapcraft login --with /snapcraft/config &>/dev/null && " +
                                        "snapcraft release $snap $edge_rev $beta_track\"")
                                }
                            }
                        }
                    }
                }
            }
        }
        catch(Exception ex) {
            error = ex
            currentBuild.result = 'FAILURE'
            println("Caught error: "+ex)
        }
        finally {
            println("Entered finally block")
            if ( params.DO_INSTALL && server_id != null) {
                delete_vm = true
                if (error && params.SAVE_CONTAINER_ON_FAIL ) {
                    delete_vm = false
                }
                if (!error && params.SAVE_CONTAINER_ON_PASS ) {
                    delete_vm = false
                }

                if ( delete_vm ) {
                    if (server_id != null) {
                        println("Deleting VM: $server_id")
                        sh """#!/bin/sh -e
                            for line in `grep OS ~/hive/robot-systest.cfg | grep -v OS_CLOUD` ; do export \$line ; done
                            openstack server delete ${server_id}
                        """
                    } else {
                        println("Saved VM $server_id in ETSI VIM")
                    }
                }
            }
            sh "docker stop ${http_server_name} || true"
            sh "docker rm ${http_server_name} || true"
        }
    }
}