#!/usr/bin/env python3
# Copyright 2020 Canonical Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# For those usages not covered by the Apache License, Version 2.0 please
# contact: legal@canonical.com
#
# To get in touch with the maintainers, please contact:
# osm-charmers@lists.launchpad.net
##

from typing import NoReturn
import unittest
from unittest.mock import patch, PropertyMock

from charm import ZookeeperCharm
from ops.model import ActiveStatus, BlockedStatus
from ops.testing import Harness


class TestCharm(unittest.TestCase):
    """Zookeeper Charm unit tests."""

    def setUp(
        self,
    ) -> NoReturn:
        """Test setup"""
        self.harness = Harness(ZookeeperCharm)
        self.harness.set_leader(is_leader=True)
        self.config = {"log_level": "INFO", "image_pull_pulicy": "always"}
        self.harness.begin()

    def test_config_invalid_log_level(self) -> NoReturn:
        """Test invalid log_level config."""
        self.config.update({"log_level": "invalid log level"})
        self.harness.update_config(self.config)
        self.assertIsInstance(self.harness.charm.unit.status, BlockedStatus)

    def test_config_invalid_image_pull_pulicy(self) -> NoReturn:
        """Test invalid image_pull_pulicy config."""
        self.config.update({"image_pull_policy": "invalid image_pull_policy"})
        self.harness.update_config(self.config)
        self.assertIsInstance(self.harness.charm.unit.status, BlockedStatus)

    @patch("charm.ZookeeperCharm.num_units", new_callable=PropertyMock)
    def test_config_changed_no_relations(self, mock_num_units) -> NoReturn:
        """Test config changed without relations."""
        mock_num_units.return_value = 1
        self.harness.charm.on.config_changed.emit()
        self.assertNotIsInstance(self.harness.charm.unit.status, BlockedStatus)

    @patch("charm.ZookeeperCharm.num_units", new_callable=PropertyMock)
    def test_config_changed_non_leader(self, mock_num_units) -> NoReturn:
        """Test config changed without relations (non-leader)."""
        mock_num_units.return_value = 1
        self.harness.set_leader(is_leader=False)
        self.harness.charm.on.config_changed.emit()

        # Assertions
        self.assertIsInstance(self.harness.charm.unit.status, ActiveStatus)

    @patch("charm.ZookeeperCharm.num_units", new_callable=PropertyMock)
    @patch("charm.ZookeeperCharm.zookeeper_uri", new_callable=PropertyMock)
    def test_with_relations_zookeeper(
        self, mock_zookeeper_uri, mock_num_units
    ) -> NoReturn:
        "Test with relations (zookeeper)"
        mock_num_units.return_value = 1
        mock_zookeeper_uri.return_value = "zk-uri"

        # Initializing the zookeeper relation
        zookeeper_relation_id = self.harness.add_relation("zookeeper", "kafka")
        self.harness.add_relation_unit(zookeeper_relation_id, "kafka/0")
        # self.harness.update_relation_data(
        #     zookeeper_relation_id, "kafka/0", {"host": "zookeeper", "port": 9092}
        # )

        # Verifying status
        self.assertNotIsInstance(self.harness.charm.unit.status, BlockedStatus)


if __name__ == "__main__":
    unittest.main()
