#!/usr/bin/env python3
# Copyright 2020 Canonical Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# For those usages not covered by the Apache License, Version 2.0 please
# contact: legal@canonical.com
#
# To get in touch with the maintainers, please contact:
# osm-charmers@lists.launchpad.net
##

from typing import NoReturn
import unittest
from unittest.mock import patch, PropertyMock

from charm import KafkaCharm
from ops.model import ActiveStatus, BlockedStatus
from ops.testing import Harness


class TestCharm(unittest.TestCase):
    """Kafka Charm unit tests."""

    def setUp(
        self,
    ) -> NoReturn:
        """Test setup"""
        self.harness = Harness(KafkaCharm)
        self.harness.set_leader(is_leader=True)
        self.harness.begin()
        self.config = {"num_partitions": 1}
        self.harness.update_config(self.config)

    def test_config_changed_no_relations(self) -> NoReturn:
        """Test config changed without relations."""
        self.harness.charm.on.config_changed.emit()
        self.assertIsInstance(self.harness.charm.unit.status, BlockedStatus)

    def test_config_changed_non_leader(self) -> NoReturn:
        """Test config changed without relations (non-leader)."""
        self.harness.set_leader(is_leader=False)
        self.harness.charm.on.config_changed.emit()

        # Assertions
        self.assertIsInstance(self.harness.charm.unit.status, ActiveStatus)

    @patch("charm.KafkaCharm.num_units", new_callable=PropertyMock)
    def test_with_relations_kafka(self, mock_num_units) -> NoReturn:
        "Test with relations (kafka)"
        mock_num_units.return_value = 1

        # Initializing the kafka relation
        zookeeper_relation_id = self.harness.add_relation("zookeeper", "zookeeper")
        self.harness.add_relation_unit(zookeeper_relation_id, "zookeeper/0")
        self.harness.update_relation_data(
            zookeeper_relation_id, "zookeeper", {"zookeeper_uri": "zk-uri"}
        )

        # Verifying status
        self.assertNotIsInstance(self.harness.charm.unit.status, BlockedStatus)


if __name__ == "__main__":
    unittest.main()
