#!/usr/bin/env python3
# Copyright 2021 Canonical Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# For those usages not covered by the Apache License, Version 2.0 please
# contact: legal@canonical.com
#
# To get in touch with the maintainers, please contact:
# osm-charmers@lists.launchpad.net
##

# pylint: disable=E0213


import logging
import re
from typing import NoReturn, Optional

from ops.main import main
from opslib.osm.charm import CharmedOsmBase, RelationsMissing
from opslib.osm.interfaces.kafka import KafkaClient
from opslib.osm.interfaces.mongo import MongoClient
from opslib.osm.interfaces.mysql import MysqlClient
from opslib.osm.pod import (
    ContainerV3Builder,
    PodRestartPolicy,
    PodSpecV3Builder,
)
from opslib.osm.validator import ModelValidator, validator


logger = logging.getLogger(__name__)

PORT = 9999
DEFAULT_MYSQL_DATABASE = "pol"


class ConfigModel(ModelValidator):
    log_level: str
    mongodb_uri: Optional[str]
    mysql_uri: Optional[str]
    image_pull_policy: str
    debug_mode: bool
    security_context: bool

    @validator("log_level")
    def validate_log_level(cls, v):
        if v not in {"INFO", "DEBUG"}:
            raise ValueError("value must be INFO or DEBUG")
        return v

    @validator("mongoddb_uri")
    def validate_mongodb_uri(cls, v):
        if v and not v.startswith("mongodb://"):
            raise ValueError("mongodb_uri is not properly formed")
        return v

    @validator("mysql_uri")
    def validate_mysql_uri(cls, v):
        pattern = re.compile("^mysql:\/\/.*:.*@.*:\d+\/.*$")  # noqa: W605
        if v and not pattern.search(v):
            raise ValueError("mysql_uri is not properly formed")
        return v

    @validator("image_pull_policy")
    def validate_image_pull_policy(cls, v):
        values = {
            "always": "Always",
            "ifnotpresent": "IfNotPresent",
            "never": "Never",
        }
        v = v.lower()
        if v not in values.keys():
            raise ValueError("value must be always, ifnotpresent or never")
        return values[v]


class PolCharm(CharmedOsmBase):
    def __init__(self, *args) -> NoReturn:
        super().__init__(
            *args,
            oci_image="image",
            vscode_workspace=VSCODE_WORKSPACE,
        )
        if self.config.get("debug_mode"):
            self.enable_debug_mode(
                pubkey=self.config.get("debug_pubkey"),
                hostpaths={
                    "POL": {
                        "hostpath": self.config.get("debug_pol_local_path"),
                        "container-path": "/usr/lib/python3/dist-packages/osm_policy_module",
                    },
                    "osm_common": {
                        "hostpath": self.config.get("debug_common_local_path"),
                        "container-path": "/usr/lib/python3/dist-packages/osm_common",
                    },
                },
            )
        self.kafka_client = KafkaClient(self, "kafka")
        self.framework.observe(self.on["kafka"].relation_changed, self.configure_pod)
        self.framework.observe(self.on["kafka"].relation_broken, self.configure_pod)

        self.mongodb_client = MongoClient(self, "mongodb")
        self.framework.observe(self.on["mongodb"].relation_changed, self.configure_pod)
        self.framework.observe(self.on["mongodb"].relation_broken, self.configure_pod)

        self.mysql_client = MysqlClient(self, "mysql")
        self.framework.observe(self.on["mysql"].relation_changed, self.configure_pod)
        self.framework.observe(self.on["mysql"].relation_broken, self.configure_pod)

    def _check_missing_dependencies(self, config: ConfigModel):
        missing_relations = []

        if (
            self.kafka_client.is_missing_data_in_unit()
            and self.kafka_client.is_missing_data_in_app()
        ):
            missing_relations.append("kafka")
        if not config.mongodb_uri and self.mongodb_client.is_missing_data_in_unit():
            missing_relations.append("mongodb")
        if not config.mysql_uri and self.mysql_client.is_missing_data_in_unit():
            missing_relations.append("mysql")
        if missing_relations:
            raise RelationsMissing(missing_relations)

    def build_pod_spec(self, image_info):
        # Validate config
        config = ConfigModel(**dict(self.config))

        if config.mongodb_uri and not self.mongodb_client.is_missing_data_in_unit():
            raise Exception("Mongodb data cannot be provided via config and relation")
        if config.mysql_uri and not self.mysql_client.is_missing_data_in_unit():
            raise Exception("Mysql data cannot be provided via config and relation")

        # Check relations
        self._check_missing_dependencies(config)

        security_context_enabled = (
            config.security_context if not config.debug_mode else False
        )

        # Create Builder for the PodSpec
        pod_spec_builder = PodSpecV3Builder(
            enable_security_context=security_context_enabled
        )

        # Add secrets to the pod
        mongodb_secret_name = f"{self.app.name}-mongodb-secret"
        pod_spec_builder.add_secret(
            mongodb_secret_name,
            {"uri": config.mongodb_uri or self.mongodb_client.connection_string},
        )
        mysql_secret_name = f"{self.app.name}-mysql-secret"
        pod_spec_builder.add_secret(
            mysql_secret_name,
            {
                "uri": config.mysql_uri
                or self.mysql_client.get_root_uri(DEFAULT_MYSQL_DATABASE)
            },
        )

        # Build Container
        container_builder = ContainerV3Builder(
            self.app.name,
            image_info,
            config.image_pull_policy,
            run_as_non_root=security_context_enabled,
        )
        container_builder.add_port(name=self.app.name, port=PORT)
        container_builder.add_envs(
            {
                # General configuration
                "ALLOW_ANONYMOUS_LOGIN": "yes",
                "OSMPOL_GLOBAL_LOGLEVEL": config.log_level,
                # Kafka configuration
                "OSMPOL_MESSAGE_DRIVER": "kafka",
                "OSMPOL_MESSAGE_HOST": self.kafka_client.host,
                "OSMPOL_MESSAGE_PORT": self.kafka_client.port,
                # Database configuration
                "OSMPOL_DATABASE_DRIVER": "mongo",
            }
        )
        container_builder.add_secret_envs(
            mongodb_secret_name, {"OSMPOL_DATABASE_URI": "uri"}
        )
        container_builder.add_secret_envs(
            mysql_secret_name, {"OSMPOL_SQL_DATABASE_URI": "uri"}
        )
        container = container_builder.build()

        # Add Pod restart policy
        restart_policy = PodRestartPolicy()
        restart_policy.add_secrets(
            secret_names=(mongodb_secret_name, mysql_secret_name)
        )
        pod_spec_builder.set_restart_policy(restart_policy)

        # Add container to pod spec
        pod_spec_builder.add_container(container)

        return pod_spec_builder.build()


VSCODE_WORKSPACE = {
    "folders": [
        {"path": "/usr/lib/python3/dist-packages/osm_policy_module"},
        {"path": "/usr/lib/python3/dist-packages/osm_common"},
    ],
    "settings": {},
    "launch": {
        "version": "0.2.0",
        "configurations": [
            {
                "name": "POL",
                "type": "python",
                "request": "launch",
                "module": "osm_policy_module.cmd.policy_module_agent",
                "justMyCode": False,
            }
        ],
    },
}


if __name__ == "__main__":
    main(PolCharm)
