#!/bin/bash
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#

set +eux

# Helm chart 1.6.0 correspondes to Airflow 2.3.0
AIRFLOW_HELM_VERSION=1.6.0
PROMPUSHGW_HELM_VERSION=1.18.2

# Install Airflow helm chart
function install_airflow() {
    [ -z "${DEBUG_INSTALL}" ] || DEBUG beginning of function
    # copy airflow-values.yaml to the destination folder
    sudo mkdir -p ${OSM_HELM_WORK_DIR}
    sudo cp ${OSM_DEVOPS}/installers/helm/values/airflow-values.yaml ${OSM_HELM_WORK_DIR}
    # update airflow-values.yaml to use the right tag
    echo "Updating Helm values file helm/values/airflow-values.yaml to use defaultAirflowTag: ${OSM_DOCKER_TAG}"
    sudo sed -i "s#defaultAirflowTag:.*#defaultAirflowTag: ${OSM_DOCKER_TAG}#g" ${OSM_HELM_WORK_DIR}/airflow-values.yaml
    if ! helm -n osm status airflow 2> /dev/null ; then
        # if it does not exist, create secrets and install
        kubectl -n osm create secret generic airflow-webserver-secret --from-literal="webserver-secret-key=$(python3 -c 'import secrets; print(secrets.token_hex(16))')"
        helm repo add apache-airflow https://airflow.apache.org
        helm repo update
        helm -n osm install airflow apache-airflow/airflow -f ${OSM_HELM_WORK_DIR}/airflow-values.yaml --version ${AIRFLOW_HELM_VERSION}
    else
        # if it exists, upgrade
        helm repo update
        helm -n osm upgrade airflow apache-airflow/airflow -f ${OSM_HELM_WORK_DIR}/airflow-values.yaml --version ${AIRFLOW_HELM_VERSION}
    fi
    [ -z "${DEBUG_INSTALL}" ] || DEBUG end of function
}

# Install Prometheus Pushgateway helm chart
function install_prometheus_pushgateway() {
    [ -z "${DEBUG_INSTALL}" ] || DEBUG end of function
    if ! helm -n osm status pushgateway 2> /dev/null ; then
        # if it does not exist, install
        helm repo add prometheus-community https://prometheus-community.github.io/helm-charts
        helm repo update
        helm -n osm install pushgateway prometheus-community/prometheus-pushgateway --version ${PROMPUSHGW_HELM_VERSION}
    else
        # if it exists, upgrade
        helm repo update
        helm -n osm upgrade pushgateway prometheus-community/prometheus-pushgateway --version ${PROMPUSHGW_HELM_VERSION}
    fi
    [ -z "${DEBUG_INSTALL}" ] || DEBUG end of function
}

# main

OSM_DEVOPS="/usr/share/osm-devops"
OSM_HELM_WORK_DIR="/etc/osm/helm"
OSM_DOCKER_TAG="13"

while getopts ":D:d:t:-: " o; do
    case "${o}" in
        D)
            OSM_DEVOPS="${OPTARG}"
            ;;
        d)
            OSM_HELM_WORK_DIR="${OPTARG}"
            ;;
        t)
            OSM_DOCKER_TAG="${OPTARG}"
            ;;
        -)
            [ "${OPTARG}" == "debug" ] && DEBUG_INSTALL="y" && continue
            echo -e "Invalid option: '--$OPTARG'\n" >&2
            exit 1
            ;;
        :)
            echo "Option -$OPTARG requires an argument" >&2
            exit 1
            ;;
        \?)
            echo -e "Invalid option: '-$OPTARG'\n" >&2
            exit 1
            ;;
        *)
            exit 1
            ;;
    esac
done

source $OSM_DEVOPS/common/logging
source $OSM_DEVOPS/common/track

echo "DEBUG_INSTALL=$DEBUG_INSTALL"
echo "OSM_DEVOPS=$OSM_DEVOPS"
echo "OSM_DOCKER_TAG=$OSM_DOCKER_TAG"
echo "OSM_HELM_WORK_DIR=$OSM_HELM_WORK_DIR"

install_airflow
track deploy_osm airflow_ok
install_prometheus_pushgateway
track deploy_osm pushgateway_ok

