# How to Set Up Virtual Infrastructure Managers (VIMs)

## OpenStack

### How to Prepare your OpenStack to be used by OSM

This sections enumerates the steps to make an OpenStack accessible for its use from OSM.

#### 1. Guarantee that OpenStack API endpoints are reachable from OSM

(particularly, it should be reachable from RO container)

#### 2. Create a management network, with DHCP enabled, reachable from OSM

You need to create a management network, with DHCP enabled, and guarantee that this management network is reachable from OSM. The network is used by the VCA (Juju) for configuring the VNFs once they are running. It is recommended to create a provider network, isolated from OpenStack. For instance, in order to create a provider network using physical interface em1 and VLAN 500 and with CIDR 10.208.0.0/24, you should run the following commands:

```bash
neutron net-create mgmt --provider:network_type=vlan --provider:physical_network=physnet_em1 --provider:segmentation_id=500 --shared
neutron subnet-create --name subnet-mgmt mgmt 10.208.0.0/24 --allocation-pool start=10.208.0.2,end=10.208.0.254
```

#### 3. Create a valid tenant/user

You need to create a tenant/user with rights to create/delete flavors. The easiest way is to create a user and assign it the role `admin`.

Another option is to change the general flavor management policies (usually at config file `/etc/nova/policy.json`) to allow flavor creation per user.

#### 4. Modify default security group or create a new one

By default OpenStack applies the `default` security group that blocks any incoming traffic to the VM. However, `ssh` access might be needed by VCA.

Therefore, you will need to modify the default security group to allow TCP port 22, or create a new security group and configure OSM to use this security group when the VIM target is addeed (see below [Adding an OpenStack VIM target to OSM](#adding-an-openstack-vim-target-to-osm)).

#### 5. Remember to upload any images required (optional)

For the time being, it is required to upload the images of the VNFs to your VIMs, so that they are available before an instantiation.

This can happen anytime, and **it is an optional step during the preparation phase, but you should always make sure that all the images that are required for a given NS/NSI are available in the VIM target before instantiation** (otherwise, OSM will throw the corresponding error message).

In the case of OpenStack, you would typically use a variant of the following command to upload an image:

```bash
openstack image create --file="./cirros-0.3.4-x86_64-disk.img" --container-format=bare --disk-format=qcow2 --public cirros034
```

### How to Add an OpenStack VIM target to OSM

Here is an example on how to use the OSM Client to add an OpenStack VIM:

```bash
osm vim-create --name openstack-site --user admin --password userpwd --auth_url http://10.10.10.11:5000/v2.0 --tenant admin --account_type openstack --config='{security_groups: default, keypair: mykey}'
```

As it can be seen above, there is a parameter called **`--config`** used to supply general configuration options, which can be used when creating the VIM target using the OSM client.

#### Configuration Options Reference

- `management_network_id`, `management_network_name`: VIM management network id/name to use for the management VLD of NS descriptors. By default it uses same vim network name as VLD name. It can be set also at instantiation time.
- `security_groups`: To be used for the deployment
- `availability_zone`: To be used for the deployment. It can be:
  - a single availability zone (all deployments will land in that zone), e.g. `availability_zone: controller`
  - several availability zones, which enables affinity and anti-affinity deployments, e.g. `availability_zone: [zone1, zone2]`
- `neutron_availability_zone_hints`: To be used for the deployment. It can be:
  - a single availability zone, e.g. `neutron_availability_zone_hints: controller`
  - several availability zones, e.g. `neutron_availability_zone_hints: [zone1, zone2]`
- `storage_availability_zone`: To be used for the deployment. It can only be a single availability zone, e.g. `storage_availability_zone: controller`
- `region_name`: The region where the VM will be deployed.
- `insecure`: (By default false). When true it allows authorization over a non trusted certificate over https
- `ca_cert`: (incompatible with insecure). root certificate file to use for validating the OpenStack certificate
- `use_existing_flavors`: (By default, `False`). Set to `True` to use the closer flavor with enough resources instead of creating a new flavor with the exact requirements. This option does not work for EPA (cpu pinning, huge pages, ...) where RO still tries to create a flavor with the needed extra expects. Use this options when you do not have admin credentials (Available from future v2.0.2 version)
- `vim_type`: Set to "VIO" to use VMware Integrated OpenStack as VIM
- `use_internal_endpoint`: Set to True to force using internal endpoints

For OpenStack API v3, the following parameters are required:

- `project_domain_id`, `project_domain_name`: If not provided, `default` is used for `project_domain_id`
- `user_domain_id`, `user_domain_name`: If not provided, `default` is used for `user_domain_id`
- `APIversion`. Only required if the auth-url URL does not end with `v3`. Set it to "v3.3" or "3" to use this OpenStack API version.

ADVANCED configuration:

- `keypair`: To be added in addition to the keypair allocated on the VNF descriptor. Provide the name of an openstack keypair
- `dataplane_physical_net`: The physical network label used in Openstack both to identify SRIOV and passthrough interfaces (nova configuration) and also to specify the VLAN ranges used by SR-IOV interfaces (neutron configuration). In Openstack, this is typically configured in `network_vlan_ranges` param as part of neutron ML2 configuration. In case of VMware Integrated Openstack (VIO), it is the ID of the distributed virtual switch.
- `use_floating_ip`: (By default false). When boolean true, a management interface of a VNFD is automatically assigned a floating\_ip -if possible-. If there is not any floating ip avalialble, it tries to get one if there is only one pool. You can use a string `pool_id` (public network id) instead of boolean true, to indicate the pool to use in case there are several. Note that deployment is not aborted if failing to asign a floating ip. The preferred method instead of this is to use a provider network
- `dataplane_net_vlan_range`: In case of VMware Integrated Openstack (VIO) provide vlan ranges for the SRIOV (binding direct) networks in format ['start\_ID - end\_ID']
- `microversion`: This is an Openstack-only parameter that allows to specify a specific microversion to be used in nova. When using `microversion: 2.32`, it enables the use of Virtual Device Role Tagging, which allows to identify each VM interface with a tag (the tag will be the name of the interface in the VNFD) and convey that information to the VM as metadata. This implementation approach is due to the warning message in <https://developer.openstack.org/api-guide/compute/microversions.html> where it is stated that microversion backwards compatibility is not guaranteed and clients should always require a specific microversion. This functionality was introduced would not work with Openstack versions previous to Newton.
- `no_port_security_extension`: Use for those OpenStack that do not have the port\_security\_extension. This property allows neutron to disable the port security. If this option is set, port security will never be disabled regardless of the descriptor indication.
- `disable_network_port_security`: Use for those OpenStack VIMs that do not support port security enabled at network level (although port\_security\_extension is present). This property allows neutron to disable the port security by default at the time of network creation for all ports created on the created networks.

The content of the config is a text formatted in yaml. The recommendation is to use a comma separated list between curly brackets `{}` and quotes `""`.

#### How to add a VMware Integrated OpenStack (VIO) as VIM target in OSM

Although VIO works as a regular OpenStack in practice, additional options may be needed to add a VIO VIM to OSM, so that OSM has all the credentials that it needs.

For instance, you can use the following command, which includes some extra parameters, to add a VIM target with VIO (e.g. site name: `openstack-site-vio4`, IP address: `10.10.10.12`, VIM tenant: `admin`, user: `admin`, password: `passwd`)

```bash
osm vim-create --name VIO --user admin --password passwd --auth_url https://10.10.10.12:5000/v3 --tenant admin \
    --account_type openstack --config '{use_floating_ip: True, insecure: true, vim_type: VIO, APIversion: v3.3,
    dataplane_physical_net: dvs-46, "use_internal_endpoint":true,"dataplane_net_vlan_range":["31-35","37-39"]}'
```

With respect to a regular Openstack, these additional configuration parameteres are added:

- `vim_type`: Set to `VIO` to use VMware Integrated openstack as VIM.
- `use_internal_endpoint`: When `true` it allows use of private API endpoints.
- `dataplane_physical_net`: The ID of the distributed virtual switch e.g 'dvs-46' in above configuration.
- `dataplane_net_vlan_range`: In case of VMware Integrated Openstack (VIO) provide vlan ranges for the SRIOV (binding direct) networks in format `['start_ID - end_ID']`.
- `vrops_site`: URI of vRealize Operations (vROPS) endpoint used by OSM to get infrastructure metrics.
- `vrops_user`: user of vRealize Operations (vROPS) used by OSM to authenticate against vROPS to read infrastructure metrics.
- `vrops_password`: password of vRealize Operations (vROPS) used by OSM to authenticate against vROPS to read infrastructure metrics.

For common options, you may refer to the general OpenStack Setup Guide.

## OpenVIM (Deprecated)

### How to set up an OpenVIM environment

A full step-by step guide for installing an OpenVIM environment from scratch can be found in [a specific chapter](14-openvim-installation.md).

### How to add OpenVIM as VIM target in OSM

To add and OpenVIM account as VIM target, you shoulld execute the following command, using the appropriate parameters (e.g. site name: `openvim-site`, IP address: `10.10.10.10`, VIM tenant: `osm`)

```bash
osm vim-create --name openvim-site --auth_url http://10.10.10.10:9080/openvim --account_type openvim \
   --description "Openvim site" --tenant osm --user dummy --password dummy
```

## VMware's vCloud Director (Deprecated)

### How to set up a vCloud Director environment

- In order to get the `vim-tenant_name` from vCloud Director and/or the tenant UUID, you should execute:

```bash
./vmwarecli.py -u admin -p 12345 -c vcloud_host_name -U Administrator -P 123456 -o test list vdc
```

```text
+--------------------------------------+----------+
|               vdc uuid               | vdc name |
+--------------------------------------+----------+
| 605ad9e8-04c5-402d-a3b7-0b6c1bacda75 |   test   |
| a5056f85-418c-4bfd-8041-adb0f48be9d9 |   TEF    |
+--------------------------------------+----------+
```

- In this example two VDC (tenants) are available for organization `test`

- Create default network by either using Web UI of vCloud director or `vmwarecli.py`

```bash
./vmwarecli.py -u admin -p 123456 -c vcloud_host_name -U Administrator -P 123456 -o test -v TEF create network test
```

```text
Crated new network test and uuid: bac9f9c6-6d1b-4af2-8211-b6258659dfb1
```

- View organization/dataceter.

```bash
./vmwarecli.py -u admin -p 123456 -c vcloud_host_name -U Administrator -P 123456 -o test view org test
```

```text
+--------------------------------------+----------+
|               vdc uuid               | vdc name |
+--------------------------------------+----------+
| 605ad9e8-04c5-402d-a3b7-0b6c1bacda75 |   test   |
| a5056f85-418c-4bfd-8041-adb0f48be9d9 |   TEF    |
+--------------------------------------+----------+
+--------------------------------------+-------------------------------------------+
|             network uuid             |                network name               |
+--------------------------------------+-------------------------------------------+
| f2e8a499-c3c4-411f-9cb5-38c0df7ccf8e |                  default                  |
| 0730eb83-bfda-43f9-bcbc-d3650a247015 |                    test                   |
+--------------------------------------+-------------------------------------------+
+--------------------------------------+--------------+
|             catalog uuid             | catalog name |
+--------------------------------------+--------------+
| 811d67dd-dd48-4e79-bb90-9ba2199fb340 |    cirros    |
| 147492d7-d25b-465c-8eb1-b181779f6f4c | ubuntuserver |
+--------------------------------------+--------------+
```

#### How to upload images in vCloud Director environment

If a user needs on-board image that is not a VMware compatible disk image format such as qcow. User need to convert qcow image to an OVF.

- The first step is convert qcow disk image to vmdk.

```bash
qemu-img convert -f qcow2 cirros-disk.img -O vmdk cirros-0.3.4-x86_64-disk.vmdk
```

- Second step.

- - Click "New in VMware Fusion , Vmware workstation or vCenter and create a VM from VMDK file created in step one.

- Third step
  - Adjust hardware setting for VM. For example, if target VMs should have only one vNIC delete all vNIC.
  - OSM will set up and attach vNIC based on VNF file.
  - Make sure hardware version for VM set to 11 or below.
  - Export VM as OVF and upload file to OSM.
    - Example of folder structure inside VNF directory. Each exported image placed inside ovfs directory.

```text
drwxr-xr-x   2 spyroot  staff        68 Oct  4 19:31 cirros
-rw-r--r--   1 spyroot  staff  13287936 May  7  2015 cirros-0.3.4-x86_64-disk.img
-rw-r--r--   1 spyroot  staff  21757952 Oct  4 19:38 cirros-0.3.4-x86_64-disk.vmdk
-rwxr-xr-x   1 spyroot  staff        57 Oct  4 18:58 convert.sh
drwxr-xr-x  10 spyroot  staff       340 Oct  4 07:24 examples
drwxr-xr-x   3 spyroot  staff       102 Oct  4 19:41 ovfs
-rw-r--r--   1 spyroot  staff     11251 Oct  4 07:24 vnf-template-2vm.yaml
-rw-r--r--   1 spyroot  staff      5931 Oct  4 07:24 vnf-template.yaml

bash$ ls -l ovfs/cirros/
total 25360
-rw-r--r--  1 spyroot  staff  12968960 Oct  4 19:41 cirros-disk1.vmdk
-rw-r--r--  1 spyroot  staff       125 Oct  4 19:41 cirros.mf
-rw-r--r--  1 spyroot  staff      5770 Oct  4 19:41 cirros.ovf
```

Note: You should create OVF image only once if all images of same VNF/OS share same hardware specs. The VM image used as reference VM in vCloud director. Each VM that OSM instantiates will use that image as reference.

- VNF preparation step.

If image is uploaded at vCloud, reference it using the image name at VNFD descriptor.

If not, use a path of an existing image at host where OSM is running

### How to add vCloud Director as VIM target in OSM

**Note: Be aware that vCD connector for VMware vCloud Director is no longer maintained and is offered as best-effort code with no guarantees. No bug fixes will be provided from the community on vCD connector. If you are interested in contributing and leading its evolution, please feel free to contact TSC mailing list [OSM_TECH@list.etsi.org](https://mail.google.com/mail/?view=cm&fs=1&tf=1&to=OSM_TECH@list.etsi.org). or RO MDL via Slack.**

```bash
osm vim-create --name vmware-site --user osm --password osm4u --auth_url https://10.10.10.12 --tenant vmware-tenant  --account_type vmware          --config '{admin_username: user, admin_password: passwd, orgname: organization, nsx_manager: "http://10.10.10.12", nsx_user: user, nsx_password: userpwd,"vcenter_port": port, "vcenter_user":user, "vcenter_password":password, "vcenter_ip": 10.10.10.14}'
```

There is a parameter called `--config` used to suply additional configuration:

- `orgname`: (Optional) Organization name where tenant belong to. Can be ignored if --vim-tenant-name uses <orgname: tenant>
- `admin_username`: (Mandatory)Admin user
- `admin_password`: (Mandatory) Admin password
- `nsx_manager`: (Mandatory). NSX manager host name
- `nsx_user`: (Mandatory). nsx\_user
- `nsx_password`: (Mandatory). nsx\_password
- `vcenter_port`: (Mandatory).vCenter port
- `vcenter_user`: (Mandatory) vCenter username
- `vcenter_password`: (Mandatory). vCenter password
- `vcenter_ip`: (Mandatory). vCenter IP
- `management_network_id`, `management_network_name`: VIM management network id/name to use for the management VLD of NS descriptors. By default it uses same vim network name as VLD name. It can be set also at instantiation time.

The content of config is a yaml format text. The recommendation is to use a comma separated list between curly brackets `{}` and quotes `""`, e.g.:

```bash
--config '{nsx_manager: https://10.10.10.12, nsx_user: user, nsx_password: pasword}'
```

## Amazon Web Services (AWS)

### How to set up an AWS environment

#### 1. Get `AWS_ACCESS_KEY_ID` and `AWS_SECRET_KEY` for your AWS account

Check <https://aws.amazon.com/>

AWS User-ID/Secret-key will be required at the time of creation of the data-center. These credentials need not to be updated after creation of the data-center.

#### 2. Create/get key-pairs from your AWS management console

These key-pairs will be required later for deploying a NS.

SSH key-pairs need to be specified in the VNF descriptors. This update can be done via OSM CLI as well as OSM UI. SSH key-pars can only be created using the AWS mgmt console. OSM will get updated of any changes that occur in AWS console. OSM user is required to keep record of these key-pairs, for use in later cases.

#### 3. Create a management network using AWS management console

If the user does not specify any default mgmt interface, OSM will create a default network that will be used for managing AWS instances.

Once the NS is deployed it will require a management interface (subnet) to apply configurations on the instances. User can manually create this mgmt interface using AWS console or leave it for OSM connector. The procedure of creating the interface from AWS is to create a subnet by specifying the appropriate CIDR block. This subnet is required to have DHCP enabled. AWS being a public cloud is accessible from OSM. The network is used by the VCA for configuring the VNFs once they are running.

#### 4. Create a valid user

Default user in AWS has the rights to perform all operations of AWS instances, subnets, VPCs, key-pairs, etc. In case, you want to create a secondary user with limited set of rights, you can use AWS mgmt console. NOTE: Each user in AWS has a separate access-Key/secret-key which must be kept secure else new credentials must be generated. The preferred way is to create a user and assign it the role "admin". Another option is ensure that the user has all the rights required to operate in AWS environment.

#### 5. Find and select images

AWS has a repository with many images available to be used for instances. In case you require to create a custom image, you can use AWS console and create your own images. In case you decide to use a pre-built image you will need to specify full mage path in VNF descriptor.

#### 6. Security group

AWS provides a default `security_group` defining a set of rules that allow connection access to the instances that have this `security_group`. In case, you require a new `security_group` you can create a new group defining the conditions that are required by your use case.

Default `security_group` doesn't allow user to SSH into the instances. This behavior is not recommended by OSM, as VCA requires a path to interact with instances. Hence, it is recommended that you create a new group that contains the rules/conditions required to SSH into the instances deployed by this NS. You can also modify the default security group to allow TCP port 22 or, however, creation of a custom security\_group is recommended.

### How to add AWS as VIM target in OSM

You will need to specify some options at VIM target creation/association by using the `--config` parameter. For instance:

```bash
osm vim-create --name aws-site --account_type aws \
   --auth_url https://aws.amazon.com \
   --user MyUser --password MyPassword --tenant admin \
   --description "AWS site, with your user" \
   --config '{region_name: eu-central-1, flavor_info: "{t2.nano: {cpus: 1, disk: 100, ram: 512}, t2.micro: {cpus: 1, disk: 100, ram: 1024}, t2.small: {cpus: 1, disk: 100, ram: 2048}}"}'
```

The following configuration can be added:

- `management_network_id`, `management_network_name`: VIM management network id/name to use for the management VLD of NS descriptors. By default it uses same vim network name as VLD name. It can be set also at instantiation time.
- `region_name`: Region to be used for the deployment
- `vpc_cidr_block`: Default CIDR block for VPC
- `security_groups`: Default security group for newly created instances

ADVANCED configuration:

- `key_pair`: Key\_pair specified here will be used default key\_pair for newly created instances
- `flavor_info`: AWS doesn't provide a mechanism to extract information regarding supported flavors. In order to get flavor information, user must specify a YAML file with the path such as: "@/usr/data/flavour\_info.yaml" or user can specify a dictionary containing details of flavors to be used.

For specification of flavor info at time of datacenter creation use a parameter at `--config` called e.g. `flavor_info:`. The content must be a string. It can be a file starting with '@' that contains the info in YAML format, or directly the yaml content.

**NOTE:** Details on AWS flavors/instance types can be found at Amazon Web Services docs (<https://aws.amazon.com/ec2/instance-types/>). Flavors/instance types in AWS vary depending on the region of AWS account. Above mentioned link provides details on all possible instance types. However to get details on the instance types available for your region, use your AWS management console.

## Microsoft Azure

### How to set up an Azure environment

#### 1. Obtain Azure credentials and tenant from Microsoft Azure

In order to use a VIM target based on Azure, the following information needs to be gathered:

- Azure `subscription Id`.
- The `tenant Id`, to be created or obtained in the Microsoft portal.
- Access credentials:
  - `client_id`: This attribute corresponds to the `application Id` of a service principal with the right permissions. Regular Azure users currently are not supported.
  - The `authentication Key`.
- Resource Group: All Azure resources for a VIM target will be created into the same `resource_group`. Said resource group could exist before or, in case it did not exist, OSM would try to create it upon the creation of the VIM target (the latter would require that your user/service principal had sufficient privileges in the subscription - check with your administrator).

The above data can be learned through the Azure portal or the Azure CLI. For instance, once logged in, the command `az account list` returns, among other information, the `application Id`, the `tenant Id`, and the `subscription Id`.

#### 2. Create Microsoft Azure Virtual Network

The virtual networks created for the Azure VIM will all be created as subnets from a base virtual network `vnet_name`. This base virtual network should be created before adding the VIM target and will also be provided as a configuration parameter.

It is also recommended to create a management subnet in that virtual network for the VIM network services, and both vnet and subnet can be created in a single step. For example, this command creates in `osmRG` resource group (must exist) the `osm_vnet` virtual network and the `internal` management subnet which uses `10.0.0.0/24` address prefix:
 
```bash
az network vnet create --resource-group osmRG --name osm_net --subnet-name internal --subnet-prefixes "10.0.0.0/24"
```

### How to add Microsoft Azure as VIM target in OSM

To sum up, in order to defice a VIM target with Azure, the following command and options should be used:

```bash
osm vim-create --name azure --account_type azure --auth_url http://www.azure.com --tenant "tenant_id"
    --user "client_id" --password "client_secret" --description "Azure site"
    --config "{region_name: westeurope, resource_group: 'osmRG', subscription_id: 'subscription_id',
     vnet_name: 'osm_vnet', flavors_pattern: 'flavors_regex'}"
```

where:

- `client_id`: Corresponds to the service principal's `application id`.
- `client secret`: Authentication key.
- `subscription_id`: Azure `subscription Id`
- `tenant`: Azure `tenant Id`

Additional required configuration:

- `region_name`: Region to be used for the deployment
- `resource_group`: Resource group to be used as base for all resources created for this VIM
- `vnet_name`: Virtual name used as a base for this VIM subnets

Additional optional configuration:

- `flavors_pattern`: Regular expression to be used during flavor selection. This allows to select the desired virtual machine tier. Microsoft Azure has a number of pre-created flavors available that cannot be changed. Hence, OSM will determine the flavor to be used based of the VDU requirements in the package, in terms of number of CPUs, RAM and disk. In the Azure portal there are also different virtual machine tiers available, intended for different purposes: e.g cheaper machine serie `Basic` with no guaranteed throughput or more expensive machines with guaranteed throughput. For that reason, OSM allows to specificy such machine tiers in the VIM target definition by using the `flavors_pattern` parameter. For example, a `Basic` cheaper tier can be selected when defining the VIM target of a development environment, and specify a more advanced tier for the VIM target of the production environment.

**NOTE for VNF Onboarding:** You need to make sure that your VNF packages include a reference to an appropriate alternative image in Microsoft Azure's image repository. In case you are creating a VNF Package from scratch, please note you should use the full Azure image name: `publisher:offer:sku:version` (e.g. `Canonical:UbuntuServer:18.04-LTS:18.04.201809110`).

## Google Cloud Platform

### How to set up a GCP environment

#### 1. Obtain GCP credentials and project from Google GCP

In order to use a VIM target based on Google Cloud, the following information needs to be gathered:

- Access credentials file (`creds`): the JSON key file corresponding to a GCP service account. If no service account is available, a new service account will have to be created under `IAM > Service accounts`. Then, once created, you can go to `Service accounts > Keys` and create a new key in JSON format for the service account. The JSON key file will be automatically downloaded.
- The `project`, the name of the Google Cloud project to be used to deploy IaaS resources (VM, nets, etc.).
- The `region`to be used to deploy IaaS resources (VM, nets, etc.).

The `project` and `region` can be obtained through the GCP portal or the GCP CLI. For instance, the command `gcloud config list` returns all properties of the active configuration, including the `account` used to authorize access to the GCP, the `project` and the `region`.

#### 2. Create Google GCP Virtual Network

You need to create a management network, with DHCP enabled, and guarantee that this management network is reachable from OSM. The network is used by the VCA (Juju) for configuring the VNFs once they are running. It is recommended to create a provider network, isolated from OpenStack. For instance, these commands creates the `osm_vnet` virtual network and the `internal` management subnet which uses `10.10.0.0/24` address prefix:

```bash
gcloud compute networks create osm-net --subnet-mode=custom
gcloud compute networks subnets create internal --network=osm-net --range=10.10.0.0/24 --region=region_id
```

### How to add GCP as VIM target in OSM

To sum up, in order to register a GCP VIM target in OSM, the following command and options should be used:

```bash
osm vim-create --name gcp --account_type gcp --project "<project>" --description "Google Cloud"
    --config "{region_name: <region>, use_floating_ip: True}"
    --creds <credential_json_file>
```

where:

- `project`: corresponds to the project name.
- `region`: corresponds to the region name.
- `credentials_json_file`: Service account JSON file.
- `use_floatinp_ip` is set to True in order to allow automatic public IP addresses in all the interfaces connected to the management network of a NS.

**NOTE for VNF Onboarding:** You need to make sure that your VNF packages include a reference to an appropriate alternative image in Google Cloud's image repository. In Google Cloud there are a set of public preconfigured images grouped in image projects. In case you are creating a VNF Package from scratch, please note you should use the following naming conventions to get the Google Cloud image:

- `<image_project>:image-family:<family>`: to get the latest version of the images of the family `<family>` in the project `<image_project>` (e.g. `ubuntu-os-cloud:image-family:ubuntu-1804-lts`)
- `<image_project>:image:<image>`: to get a specific image `<image>` in the project `<image_project>` (e.g. `ubuntu-os-cloud:image:ubuntu-1804-bionic-v20210928`)
- `<image_project>:<image>`: equivalent to `<image_project>:image:<image>` (e.g. `ubuntu-os-cloud:ubuntu-1804-bionic-v20210928`)


## Reference: Sandboxes 

Sometimes, casual testers of OSM may not have a fully fledged VIM at hand, either on premises or in a public cloud. In those cases, it is also possible using a VIM sandbox which, although limited and hence not appropriate for production services, it can be an option good enough for beginners.

In the coming sections, a few options are described in detail.

### ETSI OSM VNF Onboarding Sandbox for VNF Providers

ETSI OSM can provide to their members/participants a VNF Onboarding Sandbox for VNF Providers, based on its own testing infrastructure.

In practive, it will work as an account in a fully fledged VIM, so this alternative presents the advantage that it does not present the functional limitations that local sandboxes usually have.

More details about this option are in ellaboration and will be shared soon.

### DevStack

DevStack is a series of extensible scripts used to quickly bring up a complete OpenStack environment based on the latest versions of everything from git master. It is used interactively as a development environment and as the basis for much of the OpenStack project’s functional testing.

The OpenStack Community provides fairly detailed [documentation on DevStack](https://docs.openstack.org/devstack/latest/) and its different configurations.

Due to its simplicity, a configuration particularly interesting for running a sandbox with limited resources (e.g. in a laptop) is the [All-In-One Single Machine installation](https://docs.openstack.org/devstack/latest/guides/single-machine.html).

#### Known limitations of DevStack

TODO: Under elaboration.

### MicroStack

MicroStack is a single-machine OpenStack cloud sandbox, developed by Canonical, and deployable with a single snap packages. Currently provided OpenStack services are: Nova, Keystone, Glance, Horizon, and Neutron, which should suffice for any basic OSM testing.

Detailed documentation is available at <https://snapcraft.io/microstack> and <https://opendev.org/x/microstack>.

## Advanced setups for high I/O performance: EPA and SDN Assist

### Explanation: Overview of EPA in OSM

OSM supports EPA (Enhanced Platform Awareness) since Rel ZERO (May 2016) and usage of multiple NUMA nodes with Rel THIRTEEN (Dec 2022) which are already defined in OSM's VNF descriptors. OSM allows to utilize EPA features like mem-policy, hugepages memory, NUMA pinning, cpu thread policy, CPU pinning, setting resource quotas (CPU, memory, storage and interface). Besides, it supports the usage of passthrough and SR-IOV interfaces. Please see the [EPA Parameters](https://osm.etsi.org/docs/user-guide/latest/20-reference.html#EPA-Parameters) to check the details of supported parameters.

If your VIM supports EPA, then you don't need to do anything extra to use it from OSM. VIM connectors in OSM take advantage of EPA capabilities if the VIM supports it. All you need to do is build your descriptors and deploy.

However, ot all VIMs support EPA necessarily. To overcome this limitation, OSM has added the following two features:

- Since OSM Release ONE (October 2016), OSM includes OpenVIM as a reference VIM, with full support of EPA. You can follow the instructions in this **[link](14-openvim-installation.md)** to install and use OpenVIM.
- Since OSM Release TWO (April 2017), OSM includes a new capability in the Resource Orchestrator called **[SDN Assist](#sdn-assist)**. Through this capability, OSM can manage the dataplane underlay conectivity through an external SDN controller. The only requirement for the VIM is that it must able to use SR-IOV and/or passthrough interfaces, and expose the assigned interfaces so that the RO can use them to create the underlay connectivity. By default, the SDN Assist capability is disabled when a datacenter or VIM is added to OSM, but you can instruct OSM to enable it per VIM target.

### SDN Assist

#### Explanation: Why SDN Assist

SDN Assist works as follows to overcome the limitations of the VIM with respect to the underlay:

1. OSM deploys the VMs of a NS in the requested VIM target with Passthrough and/or SRIOV interfaces.
2. Then it retrieves from the VIM the information about the compute node where the VM was deployed and the physical interfaces assigned to the VM (identified by their PCI addressess).
3. Then, OSM maps those interfaces to the appropriate ports in the switch making use of the mapping that you should have introduced in the system.
4. Finally, OSM creates the dataplane networks by instructing the SDN controller and connecting the appropriate ports to the same network.

The module in charge of this workflow is the RO (Resource Orchestrator), which is provided transparently to the user. It uses an internal library to manage the underlay connectivity via SDN. The current library includes plugins for FloodLight, ONOS and OpenDayLight.

#### Reference: General requirements

The general requirements are:

- A dataplane switch (until Release SIX, with Openflow capabilities) that will connect the physical interfaces of the VIM compute nodes.
- An external SDN controller controlling the previous dataplane switch.
- The mapping between the switch ports (identified by name) and the compute node interfaces (identified by host-id and PCI address)
- Some VIMs as Openstack requires admin credentials in order to be able to get the physical place of the SRIOV/passthrough VM interfaces

In addition to the general requirements, every VIM will have to be properly configured.

### How to use SDN Assist

#### Adding a SDN controller to OSM

This is done through CLI.

Add to OSM  the SDN controller, provide credentials.

```bash
osm sdnc-create --name sdn-name --type arista \
--url https://10.95.134.225:443 --user osm --password osm4u \
--config '{mapping_not_needed: True, switch_id: ID}'
# The config section is optional.

# list sdn controllers with:
osm sdnc-list
# get details with:
osm sdnc-show sdn-name
# delete with
osm sdnc-delete sdn-name
```

Note that at SDN creation, connectivity and credentials are not checked.

Available SDN controller plugins (`--type` option) are `onos_vpls`, `onosof`, `floodlightof`, , `dynpac`; and comming `odlof`, `arista`, `ietfl2vpn`.

Depending on the pluging, the SDN controller needs to be fed with the mapping between the VIM compute node interfaces and the switch ports. In case this is not needed use the `--config '{mapping_not_needed: True}'`.

#### Associate the SDN controller to a VIM

To associate the SDN controller to a concrete VIM, the VIM must be updated. In this step the compute-node/swith-port mapping is provided:

```bash
osm vim-update vim-name --sdn_controller sdn-name  --sdn_port_mapping port-mapping-file.yaml
```

This is an example of the `port-mapping-file.yaml` content:

```yaml
- compute_node: nfv54
  ports:
  - pci: "0000:5d:00.1"
    switch_id: Leaf1
    switch_port: "Ethernet13/1"
  - pci: "0000:5d:0a.0"
    switch_id: Leaf1
    switch_port: "Ethernet13/1"
  - pci: "0000:5d:0a.1"
    switch_id: Leaf1
    switch_port: "Ethernet13/1"
  - pci: "0000:5d:00.0"
    switch_id: Leaf2
    switch_port: "Ethernet13/2"
  - pci: "0000:5d:02.0"
    switch_id: Leaf2
    switch_port: "Ethernet13/2"
  - pci: "0000:5d:02.1"
    switch_id: Leaf2
    switch_port: "Ethernet13/2"
- compute_node: nfv55
  # ...
```

NOTE: several PCI addresses can be connected to the same switch port. This is because a physical interface has several SR-IOV virtual interfaces, each one with different PCI address.

NOTE: The optional `switch_id` provided at `--config` is taken as a default if missing in the port-mapping file. This is useful if there is only one switch.

To overwrite the port mapping the same instruction can be used after modifying the port-mapping file.

You can check the associated SDN controller by:

```bash
osm vim-show vim-name
+-----------------+-----------------------------------------------------------------+
| key             | attribute                                                       |
+-----------------+-----------------------------------------------------------------+
| _id             | "bf900941-a6d3-4eba-9017-0cec657b9490"                          |
| name            | "sdn-name"                                                      |
| vim_type        | "openstack"                                                     |
| description     | "some description"                                              |
| vim_url         | "https://192.168.1.1:5000/v3"                                   |
| vim_user        | "osm"                                                           |
| vim_password    | "********"                                                      |
| vim_tenant_name | "osm"                                                           |
| config          | {                                                               |
|                 |   "insecure": true,                                             |
|                 |   "sdn-controller": "82b067c8-cb9a-481d-b311-59f68b75acae",     |
|                 |   "sdn-port-mapping": [                                         |
|                 |     {                                                           |
|                 |       "compute_node": "compute-66e153a8-c45",                   |
|                 |       "ports": [                                                |
|                 |         {                                                       |
|                 |           "pci": "002f-0000:83:00.0-000",                       |
|                 |           "switch_id": "Leaf1",                                 |
|                 |           "switch_port": "Ethernet52/1"                         |
```

You can disassociatede SDN controller from a VIM by:

```bash
osm vim-update vim-name --sdn_controller '' # attach an empty string
```

Note: detaching a SDN controller from the VIM is mandatory before deleting the SDN controller

### How to configure Openstack for full EPA support in OSM

Besides the instructions above for any Openstack, you should do extra configuration to configure Openstack for running VNFs which use SR-IOV interfaces.

**DO NOT consider this as the final configuration guide for Openstack with EPA. Please check Openstack docs and Openstack downstream distros' docs to get a further understanding and the up-to-date required configuration.**

**Note: The configuration shown below works with Openstack Newton, and it might not work with later versions.**

- The compute nodes need to have a whitelist for the interfaces with SRIOV and passthrough enabled, and those interface need to be associated to a physical network label e.g. `physnet`. This can be done in the file `/etc/nova/nova.conf`:

```json
pci_passthrough_whitelist=[{"devname": "p3p1", "physical_network": "physnet"}, {"devname": "p3p2", "physical_network": "physnet"}]
```

- The neutron controller needs to be updated to add `sriovnicswitch` to the `mechanism_drivers`. This can be done in the file `/etc/neutron/plugins/ml2/ml2_conf.ini`

```text
mechanism_drivers=openvswitch,sriovnicswitch
```

- The neutron controller needs to be updated to set the vlans to be used for the defined physical network label. This can be done in the file `/etc/neutron/plugins/ml2/ml2_conf.ini`. For instance, to set the vlans from 2000 to 3000:

```text
network_vlan_ranges =physnet:2000:3000
```

- The neutron controller needs to be updated to allow the supported NIC vendor’s product ID. This can be done in the file `/etc/neutron/plugins/ml2/ml2_conf.ini` to

```text
[ml2_sriov]
supported_pci_vendor_devs = 8086:10ed
```

- The nova controller needs to be updated to allow proper scheduling of SR-IOV and Passthrough devices, by adding the PciPassthroughFilter filter to the list of filters. This can be done in the file `/etc/nova/nova.conf`:

```text
scheduler_available_filters=nova.scheduler.filters.all_filters
scheduler_default_filters=RetryFilter,AvailabilityZoneFilter,RamFilter,ComputeFilter,ComputeCapabilitiesFilter,ImagePropertiesFilter,CoreFilter, PciPassthroughFilter
```

The previous configuration has taken as a reference the documents in the links below. Please check them in case your needed more details:

- <https://access.redhat.com/documentation/en-us/red_hat_openstack_platform/10/html-single/network_functions_virtualization_product_guide/>
- <https://access.redhat.com/documentation/en-us/red_hat_openstack_platform/10/html-single/network_functions_virtualization_planning_guide/>
- <https://access.redhat.com/documentation/en-us/red_hat_openstack_platform/10/html-single/network_functions_virtualization_configuration_guide/>
- <https://docs.openstack.org/newton/networking-guide/config-sriov.html>

